section {* Interpretation of syntax in semantic domains *}

(* author: Andrei Popescu *)  

theory Semantic_Domains imports Iteration (* Recursion *)
begin

text {* In this section, we employ our iteration principle
to obtain interpretation of syntax in semantic domains via valuations.
A bonus from our Horn-theoretic approach is the built-in
commutation of the interpretation with substitution versus valuation update,
a property known in the literature as the ``substitution lemma". *}

subsection {* Semantic domains and valuations *}

text{*
Semantic domains are for binding signatures
what algebras are for standard algebraic signatures.  They fix carrier sets for each sort,
and interpret each operation symbol as an operation on these sets
%
\footnote{
To match the Isabelle type system, we model (as usual) the family of carrier sets as a
``well-sortedness" predicate taking sorts and semantic items from a given
(initially unsorted) universe into booleans,
and require the operations, considered on the unsorted universe, to preserve well-sortedness.
}
%
of corresponding arity, where:
%
\\- non-binding arguments
are treated as usual (first-order) arguments;
%
\\- binding arguments are treated as second-order (functional) arguments.
%
\footnote{
In other words, syntactic bindings are captured semantically as functional bindings.}
%

In particular, for the untyped and simply-typed $\lambda$-calculi,
the semantic domains become the well-known (set-theoretic) Henkin models.

We use terminology and notation according to our general methodology employed so far:
the inhabitants of semantic domains are referred to as ``semantic items";
we prefix the reference to semantic items with an ``s": sX, sA, etc.
This convention also applies to the operations on semantic domains: ``sAbs", ``sOp", etc.

We eventually show that the function spaces consisting of maps
from valuations to semantic items form models;
in other words,
these maps can be viewed as ``generalized items"; we use for them
term-like notations ``X", ``A", etc.
(as we did in the theory that dealt with iteration).
*}

subsubsection {* Definitions: *}

datatype ('varSort,'sTerm)sAbs = sAbs 'varSort "'sTerm \<Rightarrow> 'sTerm"

record ('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom =
  sWls :: "'sort \<Rightarrow> 'sTerm \<Rightarrow> bool"
  sDummy :: "'sort \<Rightarrow> 'sTerm"
  sOp :: "'opSym \<Rightarrow> ('index,'sTerm)input \<Rightarrow> ('bindex,('varSort,'sTerm)sAbs)input \<Rightarrow>'sTerm"

text{* The type of valuations: *}

type_synonym ('varSort,'var,'sTerm)val = "'varSort \<Rightarrow> 'var \<Rightarrow> 'sTerm"

(* *************************************************** *)
context FixSyn
begin

(* A new type variable, corresponding to 'sTerm, is introduced in the context by the following
definitions and facts. *)

fun sWlsAbs where
"sWlsAbs SEM (xs,s) (sAbs xs' sF) =
 (isInBar (xs,s) \<and> xs = xs' \<and>
  (\<forall> sX. if sWls SEM (asSort xs) sX
           then sWls SEM s (sF sX)
           else sF sX = sDummy SEM s))"

definition sWlsInp where
"sWlsInp SEM delta sinp ==
 wlsOpS delta \<and> sameDom (arOf delta) sinp \<and> liftAll2 (sWls SEM) (arOf delta) sinp"

definition sWlsBinp where
"sWlsBinp SEM delta sbinp ==
 wlsOpS delta \<and> sameDom (barOf delta) sbinp \<and> liftAll2 (sWlsAbs SEM) (barOf delta) sbinp"

definition sWlsNE where
"sWlsNE SEM ==
 \<forall> s. \<exists> sX. sWls SEM s sX"

definition sWlsDisj where
"sWlsDisj SEM ==
 \<forall> s s' sX. sWls SEM s sX \<and> sWls SEM s' sX \<longrightarrow> s = s'"

definition sOpPrSWls where
"sOpPrSWls SEM ==
 \<forall> delta sinp sbinp.
   sWlsInp SEM delta sinp \<and> sWlsBinp SEM delta sbinp
   \<longrightarrow> sWls SEM (stOf delta) (sOp SEM delta sinp sbinp)"

text{* The notion of a ``well-sorted" (better read as ``well-structured")
semantic domain -- this is the actual desired notion fo a semantic domain:
%
\footnote{
As usual in Isabelle, we first define the ``raw" version,
and then ``fix" it with a well-structuredness predicate.
}
%
*}

definition wlsSEM where
"wlsSEM SEM ==
 sWlsNE SEM \<and> sWlsDisj SEM \<and> sOpPrSWls SEM"

text{* The preperties described  in the next 4 definitions turn out to be
consequences of the well-structuredness of the semantic domain: *}

definition sWlsAbsNE where
"sWlsAbsNE SEM ==
 \<forall> us s. isInBar (us,s) \<longrightarrow> (\<exists> sA. sWlsAbs SEM (us,s) sA)"

definition sWlsAbsDisj where
"sWlsAbsDisj SEM ==
 \<forall> us s us' s' sA.
   isInBar (us,s) \<and> isInBar (us',s') \<and> sWlsAbs SEM (us,s) sA \<and> sWlsAbs SEM (us',s') sA
   \<longrightarrow> us = us' \<and> s = s'"

text{* The notion of two valuations being equal everywhere but on a given variable: *}

definition eqBut where
"eqBut val val' xs x ==
 \<forall> ys y. (ys = xs \<and> y = x) \<or> val ys y = val' ys y"

definition updVal ::
"('varSort,'var,'sTerm)val \<Rightarrow>
 'var \<Rightarrow> 'sTerm \<Rightarrow> 'varSort \<Rightarrow>
 ('varSort,'var,'sTerm)val" ("_ '(_ := _')'__" 200)
where
"(val (x := sX)_xs) ==
 \<lambda> ys y. (if ys = xs \<and> y = x then sX else val ys y)"

definition swapVal ::
"'varSort \<Rightarrow> 'var \<Rightarrow> 'var \<Rightarrow> ('varSort,'var,'sTerm)val \<Rightarrow>
 ('varSort,'var,'sTerm)val"
where
"swapVal zs z1 z2 val == \<lambda>xs x. val xs (x @xs[z1 \<and> z2]_zs) "

abbreviation swapVal_abbrev ("_ ^[_ \<and> _]'__" 200) where
"val ^[z1 \<and> z2]_zs == swapVal zs z1 z2 val"

definition sWlsVal where
"sWlsVal SEM val ==
 \<forall> ys y. sWls SEM (asSort ys) (val ys y)"

(* The last argument is a dummy argument identifying the type 'var --
it may be regarded as a type argument:  *)

definition sWlsValNE ::
"('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom \<Rightarrow> 'var \<Rightarrow> bool"
where
"sWlsValNE SEM x == \<exists> (val :: ('varSort,'var,'sTerm)val). sWlsVal SEM val"

subsubsection {* Basic facts *}

lemma sWlsNE_imp_sWlsAbsNE:
assumes "sWlsNE SEM"
shows "sWlsAbsNE SEM"
unfolding sWlsAbsNE_def proof clarify
  fix xs s
  obtain sY where "sWls SEM s sY"
  using assms unfolding sWlsNE_def by auto
  moreover assume "isInBar (xs,s)"
  ultimately
  have "sWlsAbs SEM (xs,s) (sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                                           then sY
                                           else sDummy SEM s))" by simp
  thus "\<exists>sA. sWlsAbs SEM (xs,s) sA" by blast
qed

lemma sWlsDisj_imp_sWlsAbsDisj:
assumes "sWlsDisj SEM" and "sWlsNE SEM"
shows "sWlsAbsDisj SEM"
unfolding sWlsAbsDisj_def proof clarify
  fix xs s xs' s' sA
  assume wls: "sWlsAbs SEM (xs,s) sA" and wls': "sWlsAbs SEM (xs',s') sA"
  show "xs = xs' \<and> s = s'"
  proof(cases sA)
    fix us sF
    assume sA: "sA = sAbs us sF"
    hence "us = xs" using wls by simp
    moreover have "us = xs'" using sA wls' by simp
    ultimately have 1: "xs = xs'" by simp
    have "\<forall> sX. sWls SEM (asSort xs) sX \<longrightarrow> sWls SEM s (sF sX)"
    using sA wls by simp
    moreover have "\<forall> sX. sWls SEM (asSort xs) sX \<longrightarrow> sWls SEM s' (sF sX)"
    using sA wls' 1 by simp
    moreover obtain sX where "sWls SEM (asSort xs) sX"
    using assms unfolding sWlsNE_def by auto
    ultimately have "sWls SEM s (sF sX) \<and> sWls SEM s' (sF sX)" by auto
    hence "s = s'" using assms unfolding sWlsDisj_def by auto
    thus ?thesis using 1 by simp
  qed
qed

lemma sWlsNE_imp_sWlsValNE:
assumes "sWlsNE SEM"
shows "sWlsValNE SEM x"
proof-
  let ?phi = "\<lambda> ys sY. sWls SEM (asSort ys) sY"
  obtain val where val_def:
  "val = (\<lambda>ys (y::'var). SOME sY. sWls SEM (asSort ys) sY)" by blast
  show ?thesis unfolding sWlsValNE_def sWlsVal_def
  proof(rule exI[of _ val], intro conjI allI)
    fix ys y
    have "\<exists> sY. sWls SEM (asSort ys) sY"
    using assms unfolding sWlsNE_def by simp
    thus "sWls SEM (asSort ys) (val ys y)"
    unfolding val_def using someI_ex[of "?phi ys"] by simp
  qed
qed

theorem updVal_simp[simp]:
"(val (x := sX)_xs) ys y = (if ys = xs \<and> y = x then sX else val ys y)"
unfolding updVal_def by simp

theorem updVal_over[simp]:
"((val (x := sX)_xs) (x := sX')_xs) = (val (x := sX')_xs)"
unfolding updVal_def apply(rule ext)+ by auto

theorem updVal_commute:
assumes "xs \<noteq> ys \<or> x \<noteq> y"
shows "((val (x := sX)_xs) (y := sY)_ys) = ((val (y := sY)_ys) (x := sX)_xs)"
apply(rule ext)+ using assms unfolding updVal_def by auto

theorem updVal_preserves_sWls[simp]:
assumes *: "sWls SEM (asSort xs) sX" and **: "sWlsVal SEM val"
shows "sWlsVal SEM (val (x := sX)_xs)"
using assms unfolding sWlsVal_def by auto

lemmas updVal_simps =
updVal_simp updVal_over updVal_preserves_sWls

theorem swapVal_ident[simp]:
"(val ^[x \<and> x]_xs) = val"
unfolding swapVal_def by auto

theorem swapVal_compose:
"((val ^[x \<and> y]_zs) ^[x' \<and> y']_zs') =
 ((val ^[x' @zs'[x \<and> y]_zs \<and> y' @zs'[x \<and> y]_zs]_zs') ^[x \<and> y]_zs)"
unfolding swapVal_def apply(rule ext)+
proof-
  fix xs xa
  let ?u1 = "(xa @xs[x' \<and> y']_zs') @xs[x \<and> y]_zs"
  let ?u2 = "(xa @xs[x \<and> y]_zs) @xs[(x' @zs'[x \<and> y]_zs) \<and> (y' @zs'[x \<and> y]_zs)]_zs'"
  have "?u1 = ?u2" using sw_compose .
  thus "val xs ?u1 = val xs ?u2" by simp
qed

theorem swapVal_commute:
"zs \<noteq> zs' \<or> {x,y} \<inter> {x',y'} = {} \<Longrightarrow>
 ((val ^[x \<and> y]_zs) ^[x' \<and> y']_zs') = ((val ^[x' \<and> y']_zs') ^[x \<and> y]_zs)"
using swapVal_compose[of zs' x' y' zs x y val] by(auto simp add: sw_def)

lemma swapVal_involutive[simp]: "((val ^[x \<and> y]_zs) ^[x \<and> y]_zs) = val"
apply(rule ext)+ unfolding swapVal_def by auto

lemma swapVal_sym: "(val ^[x \<and> y]_zs) = (val ^[y \<and> x]_zs)"
apply(rule ext)+ unfolding swapVal_def by(auto simp add: sw_sym)

lemma swapVal_preserves_sWls1:
assumes "sWlsVal SEM val"
shows "sWlsVal SEM (val ^[z1 \<and> z2]_zs)"
using assms unfolding sWlsVal_def swapVal_def by simp

theorem swapVal_preserves_sWls[simp]:
"sWlsVal SEM (val ^[z1 \<and> z2]_zs) = sWlsVal SEM val"
using swapVal_preserves_sWls1[of _ _ zs z1 z2] proof(auto)
  assume "sWlsVal SEM (val ^[z1 \<and> z2]_zs)"
  hence "sWlsVal SEM ((val ^[z1 \<and> z2]_zs) ^[z1 \<and> z2]_zs)"
  using swapVal_preserves_sWls1[of _ _ zs z1 z2] by blast
  thus "sWlsVal SEM val" by simp
qed

lemmas swapVal_simps =
swapVal_ident swapVal_involutive swapVal_preserves_sWls

lemma updVal_swapVal:
"((val (x := sX)_xs) ^[y1 \<and> y2]_ys) =
 ((val ^[y1 \<and> y2]_ys) ((x @xs[y1 \<and> y2]_ys) := sX)_xs)"
apply(rule ext)+ unfolding swapVal_def by auto

lemma updVal_preserves_eqBut:
assumes "eqBut val val' ys y"
shows "eqBut (val (x := sX)_xs) (val' (x := sX)_xs) ys y"
using assms unfolding eqBut_def updVal_def by auto

lemma updVal_eqBut_eq:
assumes "eqBut val val' ys y"
shows "(val (y := sY)_ys) = (val' (y := sY)_ys)"
proof(rule ext)+
  fix xs x
  show "(val (y := sY)_ys) xs x = (val' (y := sY)_ys) xs x"
  proof(cases "xs = ys \<and> x = y", simp)
    assume Case: "\<not> (xs = ys \<and> x = y)"
    moreover have "val xs x = val' xs x"
    using Case assms unfolding eqBut_def by auto
    ultimately show ?thesis by auto
  qed
qed

lemma swapVal_preserves_eqBut:
assumes "eqBut val val' xs x"
shows "eqBut (val ^[z1 \<and> z2]_zs) (val' ^[z1 \<and> z2]_zs) xs (x @xs[z1 \<and> z2]_zs)"
using assms unfolding eqBut_def swapVal_def by force

subsection {* Interpretation maps *}

text{* An interpretation map, of syntax in a semantic domain,
is the usual one w.r.t. valuations.  Here we state its compostionality conditions
(including the ``substitution lemma"), and later we prove the existence of a map
satisfying these conditions.  *}

subsubsection {* Definitions *}

text {* Below, prefix ``pr" means ``preserves". *}

definition prWls where
"prWls g SEM ==
 \<forall> s X val.
    wls s X \<and> sWlsVal SEM val
    \<longrightarrow> sWls SEM s (g X val)"

definition prWlsAbs where
"prWlsAbs gA SEM ==
 \<forall> us s A val.
    wlsAbs (us,s) A \<and> sWlsVal SEM val
    \<longrightarrow> sWlsAbs SEM (us,s) (gA A val)"

definition prWlsAll where
"prWlsAll g gA SEM ==
 prWls g SEM \<and> prWlsAbs gA SEM"

definition prVar where
"prVar g SEM ==
 \<forall> xs x val.
    sWlsVal SEM val
    \<longrightarrow> g (Var xs x) val = val xs x"

definition prAbs where
"prAbs g gA SEM ==
 \<forall> xs s x X val.
    isInBar (xs,s) \<and> wls s X \<and> sWlsVal SEM val
    \<longrightarrow>
    gA (Abs xs x X) val =
    sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX then g X (val (x := sX)_xs)
                                             else sDummy SEM s)"

definition prOp where
"prOp g gA SEM ==
 \<forall> delta inp binp val.
    wlsInp delta inp \<and> wlsBinp delta binp \<and> sWlsVal SEM val
    \<longrightarrow>
    g (Op delta inp binp) val =
    sOp SEM delta (lift (\<lambda>X. g X val) inp)
                  (lift (\<lambda>A. gA A val) binp)"

definition prCons where
"prCons g gA SEM ==
 prVar g SEM \<and> prAbs g gA SEM \<and> prOp g gA SEM"

definition prFresh where
"prFresh g SEM ==
 \<forall> ys y s X val val'.
   wls s X \<and> fresh ys y X \<and>
   sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
   \<longrightarrow> g X val = g X val'"

definition prFreshAbs where
"prFreshAbs gA SEM ==
 \<forall> ys y us s A val val'.
   wlsAbs (us,s) A \<and> freshAbs ys y A \<and>
   sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
   \<longrightarrow> gA A val = gA A val'"

definition prFreshAll where
"prFreshAll g gA SEM ==
 prFresh g SEM \<and> prFreshAbs gA SEM"

definition prSwap where
"prSwap g SEM ==
 \<forall> zs z1 z2 s X val.
   wls s X \<and> sWlsVal SEM val
   \<longrightarrow>
   g (X #[z1 \<and> z2]_zs) val =
   g X (val ^[z1 \<and> z2]_zs)"

definition prSwapAbs where
"prSwapAbs gA SEM ==
 \<forall> zs z1 z2 us s A val.
   wlsAbs (us,s) A \<and> sWlsVal SEM val
   \<longrightarrow>
   gA (A $[z1 \<and> z2]_zs) val =
   gA A (val ^[z1 \<and> z2]_zs)"

definition prSwapAll where
"prSwapAll g gA SEM ==
 prSwap g SEM \<and> prSwapAbs gA SEM"

definition prSubst where
"prSubst g SEM ==
 \<forall> ys Y y s X val.
    wls (asSort ys) Y \<and> wls s X
    \<and> sWlsVal SEM val
    \<longrightarrow>
    g (X #[Y / y]_ys) val =
    g X (val (y := g Y val)_ys)"

definition prSubstAbs where
"prSubstAbs g gA SEM ==
 \<forall> ys Y y us s A val.
    wls (asSort ys) Y \<and> wlsAbs (us,s) A
    \<and> sWlsVal SEM val
    \<longrightarrow>
    gA (A $[Y / y]_ys) val =
    gA A (val (y := g Y val)_ys)"

definition prSubstAll where
"prSubstAll g gA SEM ==
 prSubst g SEM \<and> prSubstAbs g gA SEM"

definition compInt where
"compInt g gA SEM ==
 prWlsAll g gA SEM \<and> prCons g gA SEM \<and>
 prFreshAll g gA SEM \<and> prSwapAll g gA SEM \<and> prSubstAll g gA SEM"

subsubsection {* Extension of domain preservation to inputs *}

lemma prWls_wlsInp:
assumes *: "wlsInp delta inp"
and **: "prWls g SEM" and val: "sWlsVal SEM val"
shows "sWlsInp SEM delta (lift (\<lambda> X. g X val) inp)"
using assms unfolding sWlsInp_def wlsInp_iff apply auto
unfolding liftAll2_def apply auto
proof-
  fix i s sX   let ?h = "\<lambda> X. g X val"
  assume delta: "arOf delta i = Some s" and "lift ?h inp i = Some sX"
  then obtain X where sX: "sX = ?h X" and inp: "inp i = Some X"
  unfolding lift_def by (cases "inp i", auto)
  hence "wls s X" using delta * unfolding wlsInp_iff liftAll2_def by blast
  thus "sWls SEM s sX"
  unfolding sX using ** val unfolding prWls_def by simp
qed

lemma prWlsAbs_wlsBinp:
assumes *: "wlsBinp delta binp"
and **: "prWlsAbs gA SEM" and val: "sWlsVal SEM val"
shows "sWlsBinp SEM delta (lift (\<lambda> A. gA A val) binp)"
using assms unfolding sWlsBinp_def wlsBinp_iff apply auto
unfolding liftAll2_def apply auto
proof-
  fix i us s sA   let ?hA = "\<lambda> A. gA A val"
  assume delta: "barOf delta i = Some (us,s)" and "lift ?hA binp i = Some sA"
  then obtain A where sA: "sA = ?hA A" and binp: "binp i = Some A"
  unfolding lift_def by (cases "binp i", auto)
  hence "wlsAbs (us,s) A" using delta * unfolding wlsBinp_iff liftAll2_def by blast
  thus "sWlsAbs SEM (us,s) sA"
  unfolding sA using ** val unfolding prWlsAbs_def by simp
qed

end (* context FixSyn *)
(***************************************)

subsection {* The iterative model associated to a semantic domain *}

text{*
``asIMOD SEM" stands for ``SEM (regarded) as a model".
%
\footnote{
We use the word ``model" as introduced in the theory ``Models-and-Recursion".
}
%
The associated model is built essentially as follows:
%
\\- Its carrier sets consist of functions from valuations to semantic items.
%
\\- The construct operations (i.e., those corresponding to the syntactic constructs
indicated in the given binding signature) are lifted componentwise from those of the semantic domain
``SEM" (also taking into account the higher-order nature of of the semantic counterparts of abstractions).
%
\\- For a map from valuations to items (terms or abstractions), freshness of a variable ``x"
is defined as being oblivious what the argument valuation returns for ``x".
%
\\- Swapping is defined componentwise, by two iterations of the notion of swapping the
returned value of a function.
%
\\- Substitution of a semantic term ``Y" for a variable ``y" is a semantic term ``X"
is defined to map each valuation ``val" to the application of ``X" to
[``val" updated at ``y" with whatever ``Y" returns for ``val"].

Note that:
%
\\- The construct operations definitions are determined by the desired clauses of the standard
notion of interpreting syntax in a semantic domains.
%
\\- Substitution and freshness are defined having in mind the (again standard) facts of
the interpretation commuting with substitution versus valuation update and the interpretation
being oblivious to the valuation of fresh variables.
*}

subsubsection {* Definition and basic facts  *}

text{*
The next two types of ``generalized items" are used to build models from semantic domains:
%
\footnote{
Recall that ``generalized items" inhabit models.
}
%
*}

type_synonym ('varSort,'var,'sTerm)gTerm = "('varSort,'var,'sTerm)val \<Rightarrow> 'sTerm"

type_synonym ('varSort,'var,'sTerm)gAbs = "('varSort,'var,'sTerm)val \<Rightarrow> ('varSort,'sTerm)sAbs"

(* *************************************************** *)
context FixSyn
begin

definition asIMOD ::
"('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom \<Rightarrow>
 ('index,'bindex,'varSort,'sort,'opSym,'var,
  ('varSort,'var,'sTerm)gTerm,
  ('varSort,'var,'sTerm)gAbs)model"
where
"asIMOD SEM ==
 \<lparr>igWls = \<lambda>s X. \<forall> val. (sWlsVal SEM val \<or> X val = undefined) \<and>
                      (sWlsVal SEM val \<longrightarrow> sWls SEM s (X val)),
  igWlsAbs = \<lambda>(xs,s) A. \<forall> val. (sWlsVal SEM val \<or> A val = undefined) \<and>
                              (sWlsVal SEM val \<longrightarrow> sWlsAbs SEM (xs,s) (A val)),

  igVar = \<lambda>ys y. \<lambda>val. if sWlsVal SEM val then val ys y
                                           else undefined,
  igAbs =
  \<lambda>xs x X. \<lambda>val. if sWlsVal SEM val
                   then sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                                         then X (val (x := sX)_xs)
                                         else sDummy SEM (SOME s. sWls SEM s (X val)))
                   else undefined,
  igOp = \<lambda>delta inp binp. \<lambda>val.
          if sWlsVal SEM val then sOp SEM delta (lift (\<lambda>X. X val) inp)
                                                (lift (\<lambda>A. A val) binp)
                             else undefined,
  igFresh =
  \<lambda>ys y X. \<forall> val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
                       \<longrightarrow> X val = X val',
  igFreshAbs =
  \<lambda>ys y A. \<forall> val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
                       \<longrightarrow> A val = A val',
  igSwap = \<lambda>zs z1 z2 X. \<lambda>val. if sWlsVal SEM val then X (val ^[z1 \<and> z2]_zs)
                                                else undefined,
  igSwapAbs = \<lambda>zs z1 z2 A. \<lambda>val. if sWlsVal SEM val then A (val ^[z1 \<and> z2]_zs)
                                                   else undefined,
  igSubst = \<lambda>ys Y y X. \<lambda>val. if sWlsVal SEM val then X (val (y := Y val)_ys)
                                                else undefined,
  igSubstAbs = \<lambda>ys Y y A. \<lambda>val. if sWlsVal SEM val then A (val (y := Y val)_ys)
                                                   else undefined\<rparr>"

text{* Next we state, as usual, the direct definitions of the operators and relations
of associated model, freeing ourselves from
having to go through the ``asIMOD" definition each time we reason about them.  *}

lemma asIMOD_igWls:
"igWls (asIMOD SEM) s X =
 (\<forall> val. (sWlsVal SEM val \<or> X val = undefined) \<and>
         (sWlsVal SEM val \<longrightarrow> sWls SEM s (X val)))"
unfolding asIMOD_def by simp

lemma asIMOD_igWlsAbs:
"igWlsAbs (asIMOD SEM) (us,s) A =
 (\<forall> val. (sWlsVal SEM val \<or> A val = undefined) \<and>
         (sWlsVal SEM val \<longrightarrow> sWlsAbs SEM (us,s) (A val)))"
unfolding asIMOD_def by simp

lemma asIMOD_igOp:
"igOp (asIMOD SEM) delta inp binp =
(\<lambda>val. if sWlsVal SEM val then sOp SEM delta (lift (\<lambda>X. X val) inp)
                                             (lift (\<lambda>A. A val) binp)
                          else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igVar:
"igVar (asIMOD SEM) ys y =
 (\<lambda>val. if sWlsVal SEM val then val ys y else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igAbs:
"igAbs (asIMOD SEM) xs x X =
(\<lambda>val. if sWlsVal SEM val then sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                                                then X (val (x := sX)_xs)
                                                else sDummy SEM (SOME s. sWls SEM s (X val)))
                          else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igAbs2:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes *: "sWlsDisj SEM" and **: "igWls (asIMOD SEM) s X"
shows
"igAbs (asIMOD SEM) xs x X =
(\<lambda>val. if sWlsVal SEM val then sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                                                then X (val (x := sX)_xs)
                                                else sDummy SEM s)
                          else undefined)"
proof-
  {fix val :: "('varSort,'var,'sTerm)val" assume val: "sWlsVal SEM val"
   hence Xval: "sWls SEM s (X val)"
   using ** unfolding asIMOD_igWls by simp
   hence "(SOME s. sWls SEM s (X val)) = s"
   apply(rule some_equality)
   using Xval * unfolding sWlsDisj_def by auto
  }
  thus ?thesis
  unfolding asIMOD_igAbs by (simp add: ext)
qed

lemma asIMOD_igFresh:
"igFresh (asIMOD SEM) ys y X =
(\<forall> val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
             \<longrightarrow> X val = X val')"
unfolding asIMOD_def by simp

lemma asIMOD_igFreshAbs:
"igFreshAbs (asIMOD SEM) ys y A =
(\<forall> val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
             \<longrightarrow> A val = A val')"
unfolding asIMOD_def by simp

lemma asIMOD_igSwap:
"igSwap (asIMOD SEM) zs z1 z2 X =
(\<lambda>val. if sWlsVal SEM val then X (val ^[z1 \<and> z2]_zs) else undefined)"
unfolding asIMOD_def by simp+

lemma asIMOD_igSwapAbs:
"igSwapAbs (asIMOD SEM) zs z1 z2 A =
(\<lambda>val. if sWlsVal SEM val then A (val ^[z1 \<and> z2]_zs) else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igSubst:
"igSubst (asIMOD SEM) ys Y y X =
(\<lambda>val. if sWlsVal SEM val then X (val (y := Y val)_ys) else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igSubstAbs:
"igSubstAbs (asIMOD SEM) ys Y y A =
(\<lambda>val. if sWlsVal SEM val then A (val (y := Y val)_ys) else undefined)"
unfolding asIMOD_def by simp

lemma asIMOD_igWlsInp:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM"
shows
"igWlsInp (asIMOD SEM) delta inp =
 ((\<forall> val. liftAll (\<lambda>X. sWlsVal SEM val \<or> X val = undefined) inp) \<and>
  (\<forall> val. sWlsVal SEM val \<longrightarrow> sWlsInp SEM delta (lift (\<lambda>X. X val) inp)))"
proof(auto)
  fix val assume *: "igWlsInp (asIMOD SEM) delta inp"
  show "liftAll (\<lambda>X. sWlsVal SEM val \<or> X val = undefined) inp"
  unfolding liftAll_def proof(intro allI impI)
    fix i X assume inp: "inp i = Some X"
    then obtain s where "arOf delta i = Some s"
    using * unfolding igWlsInp_def sameDom_def by fastforce
    hence "igWls (asIMOD SEM) s X"
    using inp * unfolding igWlsInp_def liftAll2_def by auto
    thus "sWlsVal SEM val \<or> X val = undefined"
    unfolding asIMOD_igWls by simp
  qed
next
  fix val::"('varSort,'var,'sTerm)val"
  assume *: "igWlsInp (asIMOD SEM) delta inp" and **: "sWlsVal SEM val"
  show "sWlsInp SEM delta (lift (\<lambda>X. X val) inp)"
  unfolding sWlsInp_def using * unfolding igWlsInp_def apply simp
  unfolding liftAll2_def lift_def apply auto apply(case_tac "inp i", auto)
  proof-
    fix i s X
    assume "arOf delta i = Some s" and "inp i = Some X"
    hence "igWls (asIMOD SEM) s X"
    using * unfolding igWlsInp_def liftAll2_def by auto
    thus "sWls SEM s (X val)" using ** unfolding asIMOD_igWls by simp
  qed
next
  assume *: "\<forall>val. liftAll (\<lambda>X. sWlsVal SEM val \<or> X val = undefined) inp"
  and **: "\<forall>val. sWlsVal SEM val \<longrightarrow> sWlsInp SEM delta (lift (\<lambda>X. X val) inp)"
  show "igWlsInp (asIMOD SEM) delta inp"
  unfolding igWlsInp_def proof auto
    obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
    using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by blast
    thus "wlsOpS delta" using ** unfolding sWlsInp_def by auto
  next
    obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
    using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by blast
    thus "sameDom (arOf delta) inp" using ** unfolding sWlsInp_def by auto
  next
    show "liftAll2 (igWls (asIMOD SEM)) (arOf delta) inp"
    unfolding liftAll2_def asIMOD_igWls apply auto
    using * unfolding liftAll_def apply(case_tac "v2 val = undefined", auto)
    using ** unfolding sWlsInp_def liftAll2_def lift_def by fastforce
  qed
qed

lemma asIMOD_igSwapInp:
assumes "sWlsVal SEM val"
shows "lift (\<lambda>X. X val) (igSwapInp (asIMOD SEM) zs z1 z2 inp) =
       lift (\<lambda>X. X (swapVal zs z1 z2 val)) inp"
unfolding igSwapInp_def lift_comp
unfolding asIMOD_igSwap swapVal_def comp_def
unfolding lift_def apply(rule ext)
using assms by(case_tac "inp i", auto)

lemma asIMOD_igSubstInp:
assumes "sWlsVal SEM val"
shows "lift (\<lambda>X. X val) (igSubstInp (asIMOD SEM) ys Y y inp) =
       lift (\<lambda>X. X (val (y := Y val)_ys)) inp"
unfolding igSubstInp_def lift_comp
unfolding asIMOD_igSubst comp_def
unfolding lift_def apply(rule ext)
using assms by(case_tac "inp i", auto)

lemma asIMOD_igWlsBinp:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM"
shows
"igWlsBinp (asIMOD SEM) delta binp =
 ((\<forall> val. liftAll (\<lambda>X. sWlsVal SEM val \<or> X val = undefined) binp) \<and>
  (\<forall> val. sWlsVal SEM val \<longrightarrow> sWlsBinp SEM delta (lift (\<lambda>X. X val) binp)))"
proof(auto)
  fix val assume *: "igWlsBinp (asIMOD SEM) delta binp"
  show "liftAll (\<lambda>A. sWlsVal SEM val \<or> A val = undefined) binp"
  unfolding liftAll_def proof(intro allI impI)
    fix i A assume binp: "binp i = Some A"
    then obtain us s where "barOf delta i = Some (us,s)"
    using * unfolding igWlsBinp_def sameDom_def by force
    hence "igWlsAbs (asIMOD SEM) (us,s) A"
    using binp * unfolding igWlsBinp_def liftAll2_def by blast
    thus "sWlsVal SEM val \<or> A val = undefined"
    unfolding asIMOD_igWlsAbs by simp
  qed
next
  fix val::"('varSort,'var,'sTerm)val"
  assume *: "igWlsBinp (asIMOD SEM) delta binp" and **: "sWlsVal SEM val"
  show "sWlsBinp SEM delta (lift (\<lambda>A. A val) binp)"
  unfolding sWlsBinp_def using * unfolding igWlsBinp_def apply simp
  unfolding liftAll2_def lift_def apply auto apply(case_tac "binp i", auto)
  proof-
    fix i us s A
    assume "barOf delta i = Some (us,s)" and "binp i = Some A"
    hence "igWlsAbs (asIMOD SEM) (us,s) A"
    using * unfolding igWlsBinp_def liftAll2_def by auto
    thus "sWlsAbs SEM (us,s) (A val)" using ** unfolding asIMOD_igWlsAbs by simp
  qed
next
  assume *: "\<forall>val. liftAll (\<lambda>A. sWlsVal SEM val \<or> A val = undefined) binp"
  and **: "\<forall>val. sWlsVal SEM val \<longrightarrow> sWlsBinp SEM delta (lift (\<lambda>A. A val) binp)"
  show "igWlsBinp (asIMOD SEM) delta binp"
  unfolding igWlsBinp_def proof auto
    obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
    using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by blast
    thus "wlsOpS delta" using ** unfolding sWlsBinp_def by auto
  next
    obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
    using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by blast
    thus "sameDom (barOf delta) binp" using ** unfolding sWlsBinp_def by auto
  next
    show "liftAll2 (igWlsAbs (asIMOD SEM)) (barOf delta) binp"
    unfolding liftAll2_def apply clarify unfolding asIMOD_igWlsAbs apply auto
    using * unfolding liftAll_def apply(case_tac "v2 val = undefined", auto)
    using ** unfolding sWlsBinp_def liftAll2_def lift_def by fastforce
  qed
qed

lemma asIMOD_igSwapBinp:
assumes "sWlsVal SEM val"
shows "lift (\<lambda>A. A val) (igSwapBinp (asIMOD SEM) zs z1 z2 binp) =
       lift (\<lambda>A. A (swapVal zs z1 z2 val)) binp"
unfolding igSwapBinp_def lift_comp
unfolding asIMOD_igSwapAbs swapVal_def comp_def
unfolding lift_def apply(rule ext)
using assms by(case_tac "binp i", auto)

lemma asIMOD_igSubstBinp:
assumes "sWlsVal SEM val"
shows "lift (\<lambda>A. A val) (igSubstBinp (asIMOD SEM) ys Y y binp) =
       lift (\<lambda>A. A (val (y := Y val)_ys)) binp"
unfolding igSubstBinp_def lift_comp
unfolding asIMOD_igSubstAbs comp_def
unfolding lift_def apply(rule ext)
using assms by(case_tac "binp i", auto)

subsubsection {* The associated model is well-structured *}

text{* That is to say: it is a fresh-swap-subst
and fresh-subst-swap model (hence of course also a fresh-swap and fresh-subst) model. *}

text{* Domain disjointness: *}

lemma asIMOD_igWlsDisj:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM" and "sWlsDisj SEM"
shows "igWlsDisj (asIMOD SEM)"
unfolding igWlsDisj_def asIMOD_igWls proof auto
  fix s s' X
  assume "\<forall>(val::('varSort,'var,'sTerm)val).
             sWlsVal SEM val \<longrightarrow> sWls SEM s (X val)"
  and "\<forall>val. (sWlsVal SEM val \<or> X val = undefined) \<and>
             (sWlsVal SEM val \<longrightarrow> sWls SEM s' (X val))"
  moreover obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
  using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by auto
  ultimately have "sWls SEM s (X val) \<and> sWls SEM s' (X val)" by simp
  thus "s = s'" using assms unfolding sWlsDisj_def by auto
qed

lemma asIMOD_igWlsAbsDisj:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM" and "sWlsDisj SEM"
shows "igWlsAbsDisj (asIMOD SEM)"
unfolding igWlsAbsDisj_def asIMOD_igWlsAbs proof clarsimp
  fix us s us' s' A
  assume *: "isInBar (us, s)"  "isInBar (us', s')"
  assume "\<forall>(val::('varSort,'var,'sTerm)val).
             sWlsVal SEM val \<longrightarrow> sWlsAbs SEM (us,s) (A val)"
  and "\<forall>val. (sWlsVal SEM val \<or> A val = undefined) \<and>
             (sWlsVal SEM val \<longrightarrow> sWlsAbs SEM (us',s') (A val))"
  moreover obtain val::"('varSort,'var,'sTerm)val" where "sWlsVal SEM val"
  using assms sWlsNE_imp_sWlsValNE unfolding sWlsValNE_def by auto
  ultimately have "sWlsAbs SEM (us,s) (A val) \<and> sWlsAbs SEM (us',s') (A val)" by simp
  moreover have "sWlsAbsDisj SEM" using assms sWlsDisj_imp_sWlsAbsDisj by auto
  ultimately show "us = us' \<and> s = s'" using * unfolding sWlsAbsDisj_def by blast
qed

lemma asIMOD_igWlsAllDisj:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM" and "sWlsDisj SEM"
shows "igWlsAllDisj (asIMOD SEM)"
unfolding igWlsAllDisj_def
using assms asIMOD_igWlsDisj asIMOD_igWlsAbsDisj by auto

text {* Only ``bound arit" abstraction domains are inhabited: *}

lemma asIMOD_igWlsAbsIsInBar:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsNE SEM"
shows "igWlsAbsIsInBar (asIMOD SEM)"
proof-
  obtain val :: "('varSort,'var,'sTerm)val" where val: "sWlsVal SEM val"
  using assms sWlsNE_imp_sWlsValNE[of SEM] unfolding sWlsValNE_def by auto
  show ?thesis unfolding igWlsAbsIsInBar_def asIMOD_igWlsAbs apply auto
  proof-
    fix us s A
    assume
    "\<forall>(val::('varSort,'var,'sTerm)val).
       (sWlsVal SEM val \<or> A val = undefined) \<and>
       (sWlsVal SEM val \<longrightarrow> sWlsAbs SEM (us,s) (A val))"
    hence "sWlsAbs SEM (us,s) (A val)" using val by simp
    thus "isInBar (us,s)" by (cases "A val", simp)
  qed
qed

text{* Domain preservation by the operators *}

text{* The constructs preserve the domains: *}

lemma asIMOD_igVarIPresIGWls: "igVarIPresIGWls (asIMOD SEM)"
unfolding igVarIPresIGWls_def asIMOD_igWls asIMOD_igVar sWlsVal_def by simp

lemma asIMOD_igAbsIPresIGWls:
assumes "sWlsDisj SEM"
shows "igAbsIPresIGWls (asIMOD SEM)"
using assms
unfolding igAbsIPresIGWls_def asIMOD_igWlsAbs
apply(auto simp add: asIMOD_igAbs2)
unfolding asIMOD_igWls apply auto
by(case_tac "sWlsVal SEM val", auto)

lemma asIMOD_igOpIPresIGWls:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sOpPrSWls SEM" and "sWlsNE SEM"
shows "igOpIPresIGWls (asIMOD SEM)"
unfolding igOpIPresIGWls_def proof auto
  fix delta inp binp
  assume *: "igWlsInp (asIMOD SEM) delta inp"
            "igWlsBinp (asIMOD SEM) delta binp"
  show "igWls (asIMOD SEM) (stOf delta) (igOp (asIMOD SEM) delta inp binp)"
  unfolding asIMOD_igWls asIMOD_igOp proof auto
    fix val::"('varSort,'var,'sTerm)val" assume **: "sWlsVal SEM val"
    let ?sinp = "lift (\<lambda>X. X val) inp"
    let ?sbinp = "lift (\<lambda>A. A val) binp"
    have "sWlsInp SEM delta ?sinp \<and> sWlsBinp SEM delta ?sbinp"
    using * ** `sWlsNE SEM` asIMOD_igWlsInp asIMOD_igWlsBinp by fastforce
    thus "sWls SEM (stOf delta) (sOp SEM delta ?sinp ?sbinp)"
    using  `sOpPrSWls SEM` unfolding sOpPrSWls_def by simp
  qed
qed

lemma asIMOD_igConsIPresIGWls:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "wlsSEM SEM"
shows "igConsIPresIGWls (asIMOD SEM)"
using assms
unfolding igConsIPresIGWls_def wlsSEM_def
using
asIMOD_igVarIPresIGWls
asIMOD_igAbsIPresIGWls
asIMOD_igOpIPresIGWls
by auto

text{* Swap preserves the domains: *}

lemma asIMOD_igSwapIPresIGWls: "igSwapIPresIGWls (asIMOD SEM)"
unfolding igSwapIPresIGWls_def asIMOD_igSwap asIMOD_igWls by auto

lemma asIMOD_igSwapAbsIPresIGWlsAbs: "igSwapAbsIPresIGWlsAbs (asIMOD SEM)"
unfolding igSwapAbsIPresIGWlsAbs_def asIMOD_igSwapAbs asIMOD_igWlsAbs by auto

lemma asIMOD_igSwapAllIPresIGWlsAll: "igSwapAllIPresIGWlsAll (asIMOD SEM)"
unfolding igSwapAllIPresIGWlsAll_def
using asIMOD_igSwapIPresIGWls
asIMOD_igSwapAbsIPresIGWlsAbs by auto

text {* Subst preserves the domains:  *}

lemma asIMOD_igSubstIPresIGWls: "igSubstIPresIGWls (asIMOD SEM)"
unfolding igSubstIPresIGWls_def asIMOD_igSubst asIMOD_igWls by simp

lemma asIMOD_igSubstAbsIPresIGWlsAbs: "igSubstAbsIPresIGWlsAbs (asIMOD SEM)"
unfolding igSubstAbsIPresIGWlsAbs_def asIMOD_igSubstAbs asIMOD_igWls asIMOD_igWlsAbs by simp

lemma asIMOD_igSubstAllIPresIGWlsAll: "igSubstAllIPresIGWlsAll (asIMOD SEM)"
unfolding igSubstAllIPresIGWlsAll_def
using asIMOD_igSubstIPresIGWls
asIMOD_igSubstAbsIPresIGWlsAbs by auto

text {* The clauses for fresh hold:  *}

lemma asIMOD_igFreshIGVar: "igFreshIGVar (asIMOD SEM)"
unfolding igFreshIGVar_def asIMOD_igFresh asIMOD_igVar eqBut_def by force

lemma asIMOD_igFreshIGAbs1:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsDisj SEM"
shows "igFreshIGAbs1 (asIMOD SEM)"
using assms
unfolding igFreshIGAbs1_def asIMOD_igFresh asIMOD_igFreshAbs
apply (auto simp add: asIMOD_igAbs2)
proof(rule ext, auto)
  fix ys y s X and val val' :: "('varSort,'var,'sTerm)val" and sX
  assume  "eqBut val val' ys y"
  and "sWlsVal SEM val" and "sWlsVal SEM val'"
  hence "(val (y := sX)_ys) = (val' (y := sX)_ys)"
  by(auto simp add: updVal_eqBut_eq)
  thus "X (val (y := sX)_ys) = X (val' (y := sX)_ys)" by simp
qed

lemma asIMOD_igFreshIGAbs2:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsDisj SEM"
shows "igFreshIGAbs2 (asIMOD SEM)"
using assms
unfolding igFreshIGAbs2_def asIMOD_igFresh asIMOD_igFreshAbs
apply (auto simp add: asIMOD_igAbs2)
apply(rule ext, auto) apply(case_tac "sWls SEM (asSort xs) sX", auto)
proof-
  fix ys y xs x s X and val val' :: "('varSort,'var,'sTerm)val" and sX
  assume
  *: "\<forall>val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' ys y
                 \<longrightarrow> X val = X val'"
  and **: "eqBut val val' ys y"
  and ***: "sWlsVal SEM val"  "sWlsVal SEM val'" "sWls SEM (asSort xs) sX"
  let ?valsb = "val (x := sX)_xs"  let ?valsb' = "val' (x := sX)_xs"
  have "sWlsVal SEM ?valsb \<and> sWlsVal SEM ?valsb'" using *** by auto
  moreover have "eqBut ?valsb ?valsb' ys y"
  using ** by(simp add: updVal_preserves_eqBut)
  ultimately show "X ?valsb = X ?valsb'" using * by simp
qed

lemma asIMOD_igFreshIGOp:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
shows "igFreshIGOp (asIMOD SEM)"
unfolding igFreshIGOp_def proof clarify
  fix ys y delta and inp :: "('index, ('varSort,'var,'sTerm)gTerm)input"
  and binp :: "('bindex, ('varSort,'var,'sTerm)gAbs)input"
  assume inp_fresh: "igFreshInp (asIMOD SEM) ys y inp"
                    "igFreshBinp (asIMOD SEM) ys y binp"
  show "igFresh (asIMOD SEM) ys y (igOp (asIMOD SEM) delta inp binp)"
  unfolding asIMOD_igFresh asIMOD_igOp proof auto
    fix val val'
    let ?sinp = "lift (\<lambda>X. X val) inp" let ?sinp' = "lift (\<lambda>X. X val') inp"
    let ?sbinp = "lift (\<lambda>A. A val) binp" let ?sbinp' = "lift (\<lambda>A. A val') binp"
    assume wls: "sWlsVal SEM val" "sWlsVal SEM val'"
    and eqBut: "eqBut val val' ys y"
    have "?sinp = ?sinp'"
    unfolding lift_def proof(rule ext, case_tac "inp i", auto)
      fix i X assume "inp i = Some X"
      hence "igFresh (asIMOD SEM) ys y X"
      using inp_fresh unfolding igFreshInp_def errMOD_def liftAll_def by simp
      thus "X val = X val'" using wls eqBut unfolding asIMOD_igFresh by simp
    qed
    moreover
    have "?sbinp = ?sbinp'"
    unfolding lift_def proof(rule ext, case_tac "binp i", auto)
      fix i A assume "binp i = Some A"
      hence "igFreshAbs (asIMOD SEM) ys y A"
      using inp_fresh unfolding igFreshBinp_def errMOD_def liftAll_def by simp
      thus "A val = A val'" using wls eqBut unfolding asIMOD_igFreshAbs by simp
    qed
    ultimately
    show "sOp SEM delta ?sinp ?sbinp = sOp SEM delta ?sinp' ?sbinp'" by simp
  qed
qed

lemma asIMOD_igFreshCls:
assumes "sWlsDisj SEM"
shows "igFreshCls (asIMOD SEM)"
using assms
unfolding igFreshCls_def
using
asIMOD_igFreshIGVar
asIMOD_igFreshIGAbs1 asIMOD_igFreshIGAbs2
asIMOD_igFreshIGOp
by auto

text {* The clauses for swap hold:  *}

lemma asIMOD_igSwapIGVar: "igSwapIGVar (asIMOD SEM)"
unfolding igSwapIGVar_def apply clarsimp apply(rule ext)
unfolding asIMOD_igSwap asIMOD_igVar apply clarsimp
unfolding swapVal_def by simp

lemma asIMOD_igSwapIGAbs: "igSwapIGAbs (asIMOD SEM)"
unfolding igSwapIGAbs_def apply clarify apply(rule ext)
unfolding asIMOD_igSwap asIMOD_igSwapAbs asIMOD_igAbs apply auto
apply(rule ext) apply(case_tac "sWls SEM (asSort xs) sX", auto)
by(auto simp add: updVal_swapVal)

lemma asIMOD_igSwapIGOp:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
shows "igSwapIGOp (asIMOD SEM)"
unfolding igSwapIGOp_def apply clarify proof(rule ext)
  fix zs z1 z2 delta inp binp val
  assume inp: "igWlsInp (asIMOD SEM) delta inp"  "igWlsBinp (asIMOD SEM) delta binp"
  obtain inpsw and binpsw where
  inpsw_def: "inpsw = igSwapInp (asIMOD SEM) zs z1 z2 inp"
             "binpsw = igSwapBinp (asIMOD SEM) zs z1 z2 binp" by blast
  hence inpsw_rev: "igSwapInp (asIMOD SEM) zs z1 z2 inp = inpsw"
                   "igSwapBinp (asIMOD SEM) zs z1 z2 binp = binpsw" by auto
  let ?sinpsw = "lift (\<lambda>X. X (swapVal zs z1 z2 val)) inp"
  let ?sbinpsw = "lift (\<lambda>A. A (swapVal zs z1 z2 val)) binp"
  show "igSwap (asIMOD SEM) zs z1 z2 (igOp (asIMOD SEM) delta inp binp) val =
        igOp (asIMOD SEM) delta (igSwapInp (asIMOD SEM) zs z1 z2 inp)
                              (igSwapBinp (asIMOD SEM) zs z1 z2 binp) val"
  unfolding inpsw_rev unfolding asIMOD_igSwap asIMOD_igOp apply auto
  unfolding inpsw_def by(auto simp add: asIMOD_igSwapInp asIMOD_igSwapBinp)
qed

lemma asIMOD_igSwapCls: "igSwapCls (asIMOD SEM)"
unfolding igSwapCls_def
using
asIMOD_igSwapIGVar
asIMOD_igSwapIGAbs
asIMOD_igSwapIGOp
by auto

text{* The clauses for subst hold: *}

lemma asIMOD_igSubstIGVar1:
"igSubstIGVar1 (asIMOD SEM)"
unfolding igSubstIGVar1_def asIMOD_igSubst asIMOD_igVar asIMOD_igWls apply clarsimp
apply(rule ext)
by(case_tac "sWlsVal SEM val", auto)

lemma asIMOD_igSubstIGVar2:
"igSubstIGVar2 (asIMOD SEM)"
unfolding igSubstIGVar2_def asIMOD_igSubst asIMOD_igVar asIMOD_igWls apply clarsimp
apply(rule ext)
by(case_tac "sWlsVal SEM val", auto)

lemma asIMOD_igSubstIGAbs:
"igSubstIGAbs (asIMOD SEM)"
unfolding igSubstIGAbs_def apply clarsimp proof(rule ext)
  fix ys y Y xs x s X val
  assume Y: "igWls (asIMOD SEM) (asSort ys) Y"
  and X: "igWls (asIMOD SEM) s X" and x_diff_y: "xs = ys \<longrightarrow> x \<noteq> y"
  and x_fresh_Y: "igFresh (asIMOD SEM) xs x Y"
  show "igSubstAbs (asIMOD SEM) ys Y y (igAbs (asIMOD SEM) xs x X) val =
        igAbs (asIMOD SEM) xs x (igSubst (asIMOD SEM) ys Y y X) val"
  proof(cases "sWlsVal SEM val")
    assume "\<not> sWlsVal SEM val"
    thus ?thesis unfolding asIMOD_igSubst asIMOD_igSubstAbs asIMOD_igAbs by simp
  next
    assume val: "sWlsVal SEM val"
    hence Yval: "sWls SEM (asSort ys) (Y val)"
    using Y unfolding asIMOD_igWls by simp
    thus ?thesis
    unfolding asIMOD_igSubst asIMOD_igSubstAbs asIMOD_igAbs apply auto
    apply(rule ext) apply(case_tac "sWls SEM (asSort xs) sX")
    using val proof auto
      fix sX assume sX: "sWls SEM (asSort xs) sX"
      let ?val_x = "val (x := sX)_xs"
      have "sWlsVal SEM ?val_x" using val sX by simp
      moreover have "eqBut ?val_x val xs x"
      unfolding eqBut_def updVal_def by simp
      ultimately have 1: "Y ?val_x = Y val"
      using val x_fresh_Y unfolding asIMOD_igFresh by simp
      let ?Left = "X ((val (y := Y val)_ys) (x := sX)_xs)"
      let ?Riight = "X (?val_x (y := Y ?val_x)_ys)"
      have "?Left = X (?val_x (y := Y val)_ys)"
      using x_diff_y by(auto simp add: updVal_commute)
      also have "\<dots> = ?Riight" using 1 by simp
      finally show "?Left = ?Riight" .
    qed
  qed
qed

lemma asIMOD_igSubstIGOp:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
shows "igSubstIGOp (asIMOD SEM)"
unfolding igSubstIGOp_def apply clarify proof(rule ext)
  fix ys y Y delta inp binp val
  assume Y: "igWls (asIMOD SEM) (asSort ys) Y"
  and inp: "igWlsInp (asIMOD SEM) delta inp"
  and binp: "igWlsBinp (asIMOD SEM) delta binp"
  obtain inpsb binpsb where
  inpsb_def: "inpsb = igSubstInp (asIMOD SEM) ys Y y inp"
             "binpsb = igSubstBinp (asIMOD SEM) ys Y y binp" by blast
  hence inpsb_rev: "igSubstInp (asIMOD SEM) ys Y y inp = inpsb"
                   "igSubstBinp (asIMOD SEM) ys Y y binp = binpsb" by auto
  let ?sinpsb = "lift (\<lambda>X. X (val (y := Y val)_ys)) inp"
  let ?sbinpsb = "lift (\<lambda>A. A (val (y := Y val)_ys)) binp"
  show "igSubst (asIMOD SEM) ys Y y (igOp (asIMOD SEM) delta inp binp) val =
        igOp (asIMOD SEM) delta (igSubstInp (asIMOD SEM) ys Y y inp)
                              (igSubstBinp (asIMOD SEM) ys Y y binp) val"
  unfolding inpsb_rev unfolding asIMOD_igSubst asIMOD_igOp apply auto
  unfolding inpsb_def apply(auto simp add: asIMOD_igSubstInp asIMOD_igSubstBinp)
  using Y unfolding asIMOD_def by auto
qed

lemma asIMOD_igSubstCls:
"igSubstCls (asIMOD SEM)"
unfolding igSubstCls_def
using
asIMOD_igSubstIGVar1 asIMOD_igSubstIGVar2
asIMOD_igSubstIGAbs
asIMOD_igSubstIGOp
by auto

text {* The fresh-swap-based congruence clause holds: *}

lemma updVal_swapVal_eqBut:
"eqBut (val (x := sX)_xs) ((val (y := sX)_xs) ^[y \<and> x]_xs) xs y"
unfolding updVal_def swapVal_def eqBut_def apply(intro allI impI)
apply(case_tac "ys = xs \<and> ya = y") unfolding sw_def by auto

lemma asIMOD_igAbsCongS:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsDisj SEM"
shows "igAbsCongS (asIMOD SEM)"
using assms
unfolding igAbsCongS_def asIMOD_igFresh asIMOD_igSwap
apply (auto simp add: asIMOD_igAbs2)
apply(rule ext) apply(case_tac "sWlsVal SEM val", simp_all)
apply(rule ext) apply(case_tac "sWls SEM (asSort xs) sX", simp_all)
proof-
  fix xs x x' y s X X' and val :: "('varSort,'var,'sTerm)val" and  sX
  assume val: "sWlsVal SEM val"
  and X_ext: "\<forall>val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' xs y
                          \<longrightarrow> X val = X val'"
  and X'_ext: "\<forall>val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' xs y
                          \<longrightarrow> X' val = X' val'"
  and sX: "sWls SEM (asSort xs) sX"
  and *: "(\<lambda>val. if sWlsVal SEM val then X (val ^[y \<and> x]_xs)
               else undefined) =
          (\<lambda>val. if sWlsVal SEM val then X' (val ^[y \<and> x']_xs)
               else undefined)"
  let ?val_x = "val (x := sX)_xs"   let ?val_x' = "val (x' := sX)_xs"
  let ?val_y = "val (y := sX)_xs"
  let ?val_y_yx = "?val_y  ^[y \<and> x]_xs" let ?val_y_yx' = "?val_y  ^[y \<and> x']_xs"
  have val_x: "sWlsVal SEM ?val_x" using val sX by simp
  have val_x': "sWlsVal SEM ?val_x'" using val sX by simp
  have val_y: "sWlsVal SEM ?val_y" using val sX by simp
  hence val_y_yx: "sWlsVal SEM ?val_y_yx" using sX by simp
  have val_y_yx': "sWlsVal SEM ?val_y_yx'" using val_y sX by simp
  let ?phi = "\<lambda>x X val. if sWlsVal SEM val then X (swapVal xs y x val) else undefined"
  (*  *)
  have "eqBut ?val_x ?val_y_yx xs y" using updVal_swapVal_eqBut .
  hence "X ?val_x = X ?val_y_yx" using val_x val_y_yx X_ext by simp
  also have "\<dots> = X' ?val_y_yx'"
  using * val_y fun_cong[of "?phi x X" "?phi x' X'" ?val_y] by simp
  also
  {have "eqBut ?val_x' ?val_y_yx' xs y" using updVal_swapVal_eqBut .
   hence "X' ?val_y_yx' = X' ?val_x'" using val_x' val_y_yx' X'_ext by fastforce
  }
  finally show "X ?val_x = X' ?val_x'" .
qed

text {* The abstraction-renaming clause holds: *}

lemma asIMOD_igAbs3:
assumes "sWlsDisj SEM" and "igWls (asIMOD SEM) s X"
shows
"igAbs (asIMOD SEM) xs y (igSubst (asIMOD SEM) xs (igVar (asIMOD SEM) xs y) x X) =
 (\<lambda>val. if sWlsVal SEM val
            then sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                                 then (igSubst (asIMOD SEM) xs (igVar (asIMOD SEM) xs y) x X) (val (y := sX)_xs)
                                 else sDummy SEM s)
            else undefined)"
apply(rule asIMOD_igAbs2)
using assms asIMOD_igVarIPresIGWls asIMOD_igSubstIPresIGWls
unfolding igVarIPresIGWls_def igSubstIPresIGWls_def
apply auto by blast

lemma asIMOD_igAbsRen:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "sWlsDisj SEM"
shows "igAbsRen (asIMOD SEM)"
using assms
unfolding igAbsRen_def asIMOD_igFresh asIMOD_igSwap
apply (auto simp add: asIMOD_igAbs2 asIMOD_igAbs3) apply(rule ext, auto) apply(rule ext)
apply(case_tac "sWls SEM (asSort xs) sX", auto)
unfolding asIMOD_igSubst asIMOD_igVar apply auto
proof-
  fix xs and y x :: 'var and s X and val :: "('varSort,'var,'sTerm)val" and sX
  assume val: "sWlsVal SEM val" and sX: "sWls SEM (asSort xs) sX"
  and *: "\<forall>val val'. sWlsVal SEM val \<and> sWlsVal SEM val' \<and> eqBut val val' xs y \<longrightarrow> X val = X val'"
  let ?val_yx = "val (y := sX)_xs (x := sX)_xs"
  let ?val_x = "val (x := sX)_xs"
  have "sWlsVal SEM ?val_yx \<and> sWlsVal SEM ?val_x"
  using val sX by simp
  moreover have "eqBut ?val_yx ?val_x xs y"
  unfolding eqBut_def by simp
  ultimately show "X ?val_yx = X ?val_x" using * by simp
qed

text {* The associated model forms well-structured models of all 4 kinds: *}

lemma asIMOD_wlsFSw:
assumes "wlsSEM SEM"
shows "iwlsFSw (asIMOD SEM)"
using assms
unfolding wlsSEM_def iwlsFSw_def
using assms (* sic *)
using
asIMOD_igWlsAllDisj[of SEM] asIMOD_igWlsAbsIsInBar[of SEM]
asIMOD_igConsIPresIGWls[of SEM] asIMOD_igSwapAllIPresIGWlsAll[of SEM]
asIMOD_igFreshCls[of SEM] asIMOD_igSwapCls[of SEM]
asIMOD_igAbsCongS[of SEM]
by simp

lemma asIMOD_wlsFSb:
assumes "wlsSEM SEM"
shows "iwlsFSb (asIMOD SEM)"
using assms
unfolding wlsSEM_def iwlsFSb_def
using assms (* sic *)
using
asIMOD_igWlsAllDisj[of SEM] asIMOD_igWlsAbsIsInBar[of SEM]
asIMOD_igConsIPresIGWls[of SEM] asIMOD_igSubstAllIPresIGWlsAll[of SEM]
asIMOD_igFreshCls[of SEM] asIMOD_igSubstCls[of SEM]
asIMOD_igAbsRen[of SEM]
by auto

lemma asIMOD_wlsFSwSb: "wlsSEM SEM \<Longrightarrow> iwlsFSwSb (asIMOD SEM)"
unfolding iwlsFSwSb_def
using asIMOD_wlsFSw
asIMOD_igSubstAllIPresIGWlsAll asIMOD_igSubstCls by auto

lemma asIMOD_wlsFSbSw: "wlsSEM SEM \<Longrightarrow> iwlsFSbSw (asIMOD SEM)"
unfolding iwlsFSbSw_def
using asIMOD_wlsFSb
asIMOD_igSwapAllIPresIGWlsAll asIMOD_igSwapCls by auto

subsection {* The semantic interpretation  *}

text{* The well-definedness of the semantic interpretation, as well
as its associated substitution lemma and non-dependence of fresh variables,
are the end products of this theory.

Note that in order to establish these results either fresh-subst-swap or
fresh-swap-subst aligebras would do the job, and, moreover, if we did not care
about swapping, fresh-subst aligebras would do the job.  Therefore, our
exhaustive study of the model from previous section had a deigree of redundancy w.r.t. to our main
igoal -- we pursued it however in order to better illustrate the rich structure laying under
the apparent paucity of the notion of a semantic domain.  Next, we choose to employ
fresh-subst-swap aligebras to establish the required results. (Recall however that either aligebraic route
we take, the initial morphism turns out to be the same function.)*}

definition semInt where
"semInt SEM == iter (asIMOD SEM)"

definition semIntAbs where
"semIntAbs SEM == iterAbs (asIMOD SEM)"

lemma semIntAll_termFSwSbImorph:
"wlsSEM SEM \<Longrightarrow>
 termFSwSbImorph (semInt SEM) (semIntAbs SEM) (asIMOD SEM)"
unfolding semInt_def semInt_def semIntAbs_def
using asIMOD_wlsFSbSw[of SEM] iwlsFSbSw_iterAll_termFSwSbImorph[of "asIMOD SEM"] by simp

lemma semInt_prWls:
assumes "wlsSEM SEM"
shows "prWls (semInt SEM) SEM"
unfolding prWls_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresWlsAll_def ipresWls_def
unfolding asIMOD_igWls by auto

lemma semIntAbs_prWlsAbs:
assumes "wlsSEM SEM"
shows "prWlsAbs (semIntAbs SEM) SEM"
unfolding prWlsAbs_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresWlsAll_def ipresWlsAbs_def
unfolding asIMOD_igWlsAbs by blast

lemma semIntAll_prWlsAll:
assumes "wlsSEM SEM"
shows "prWlsAll (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding prWlsAll_def
by(simp add: semInt_prWls semIntAbs_prWlsAbs)

lemma semInt_prVar:
assumes "wlsSEM SEM"
shows "prVar (semInt SEM) SEM"
using assms semIntAll_termFSwSbImorph
unfolding prVar_def termFSwSbImorph_def termFSwImorph_def ipresCons_def ipresVar_def
unfolding asIMOD_igVar by fastforce

lemma semIntAll_prAbs:
fixes SEM :: "('index,'bindex,'varSort,'sort,'opSym,'sTerm)semDom"
assumes "wlsSEM SEM"
shows "prAbs (semInt SEM) (semIntAbs SEM) SEM"
unfolding prAbs_def apply auto
proof-
  fix xs s x X and val :: "('varSort,'var,'sTerm)val"
  assume xs_s: "isInBar (xs,s)" and X: "wls s X"
  and val: "sWlsVal SEM val"
  let ?L = "semIntAbs SEM (Abs xs x X)"
  let ?R = "\<lambda> val. sAbs xs (\<lambda>sX. if sWls SEM (asSort xs) sX
                              then semInt SEM X (val (x := sX)_xs)
                              else sDummy SEM s)"
  have "?L = igAbs (asIMOD SEM) xs x (semInt SEM X)"
  using xs_s X assms semIntAll_termFSwSbImorph[of SEM]
  unfolding termFSwSbImorph_def termFSwImorph_def ipresCons_def ipresAbs_def by auto
  moreover
  {have "prWls (semInt SEM) SEM" using assms semInt_prWls by auto
   hence 1: "sWls SEM s (semInt SEM X val)"
   using val X unfolding prWls_def by simp
   hence "(SOME s. sWls SEM s (semInt SEM X val)) = s"
   apply(rule some_equality)
   using 1 assms unfolding wlsSEM_def sWlsDisj_def by auto
   hence "igAbs (asIMOD SEM) xs x (semInt SEM X) val = ?R val"
   unfolding asIMOD_igAbs using val apply simp
   by(rule ext, auto)
  }
  ultimately show "?L val = ?R val" by simp
qed

lemma semIntAll_prOp:
assumes "wlsSEM SEM"
shows "prOp (semInt SEM) (semIntAbs SEM) SEM"
unfolding prOp_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresCons_def ipresOp_def
unfolding asIMOD_igOp
unfolding lift_comp
unfolding comp_def
by fastforce

lemma semIntAll_prCons:
assumes "wlsSEM SEM"
shows "prCons (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding prCons_def
by(simp add: semInt_prVar semIntAll_prAbs semIntAll_prOp)

lemma semInt_prFresh:
assumes "wlsSEM SEM"
shows "prFresh (semInt SEM) SEM"
unfolding prFresh_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresFreshAll_def ipresFresh_def
unfolding asIMOD_igFresh by fastforce

lemma semIntAbs_prFreshAbs:
assumes "wlsSEM SEM"
shows "prFreshAbs (semIntAbs SEM) SEM"
unfolding prFreshAbs_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresFreshAll_def ipresFreshAbs_def
unfolding asIMOD_igFreshAbs by fastforce

lemma semIntAll_prFreshAll:
assumes "wlsSEM SEM"
shows "prFreshAll (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding prFreshAll_def
by(simp add: semInt_prFresh semIntAbs_prFreshAbs)

lemma semInt_prSwap:
assumes "wlsSEM SEM"
shows "prSwap (semInt SEM) SEM"
unfolding prSwap_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresSwapAll_def ipresSwap_def
unfolding asIMOD_igSwap by fastforce

lemma semIntAbs_prSwapAbs:
assumes "wlsSEM SEM"
shows "prSwapAbs (semIntAbs SEM) SEM"
unfolding prSwapAbs_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresSwapAll_def ipresSwapAbs_def
unfolding asIMOD_igSwapAbs by fastforce

lemma semIntAll_prSwapAll:
assumes "wlsSEM SEM"
shows "prSwapAll (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding prSwapAll_def
by(simp add: semInt_prSwap semIntAbs_prSwapAbs)

lemma semInt_prSubst:
assumes "wlsSEM SEM"
shows "prSubst (semInt SEM) SEM"
unfolding prSubst_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresSubstAll_def ipresSubst_def
unfolding asIMOD_igSubst by fastforce

lemma semIntAbs_prSubstAbs:
assumes "wlsSEM SEM"
shows "prSubstAbs (semInt SEM) (semIntAbs SEM) SEM"
unfolding prSubstAbs_def
using assms semIntAll_termFSwSbImorph
unfolding termFSwSbImorph_def termFSwImorph_def ipresSubstAll_def ipresSubstAbs_def
unfolding asIMOD_igSubstAbs by fastforce

lemma semIntAll_prSubstAll:
assumes "wlsSEM SEM"
shows "prSubstAll (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding prSubstAll_def
by(simp add: semInt_prSubst semIntAbs_prSubstAbs)

theorem semIntAll_compInt:
assumes "wlsSEM SEM"
shows "compInt (semInt SEM) (semIntAbs SEM) SEM"
using assms unfolding compInt_def
by(simp add: semIntAll_prWlsAll semIntAll_prCons
semIntAll_prFreshAll semIntAll_prSwapAll semIntAll_prSubstAll)

lemmas semDom_simps =
updVal_simps swapVal_simps

end (* context FixSyn *)

end
