section {* General Recursion *}

(* author: Andrei Popescu *)

theory Recursion imports Iteration
begin

text{* The initiality theorems from the previous section support iteration principles.
Next we extend the results to general recursion.  The difference between
general recursion and iteration is that the former also considers
the (source) ``items" (terms and abstractions), and not only the
(target) generalized items, appear in the recursive clauses.
%
(Here is an example illustrating the above difference for the standard case
of natural numbers:
\\- Given a number n, the operator ``add-n" can be defined by iteration:
\\--- ``add-n 0 = n",
\\--- ``add-n (Suc m) = Suc (add-n m)".

Notice that, in right-hand side of the recursive clause, ``m" is not used ``directly", but only
via ``add-n" -- this makes the definition iterative. By contrast, the following
definition of predecessor is trivial form of recursion (namely, case analysis),
but is {\em not} iteration:
\\--- ``pred 0 = 0",
\\--- ``pred (Suc n) = n".
)

We achieve our desired extension by augmenting the notion of model
and then essentially inferring recursion (as customary)
from
[iteration having as target the product between the term model and the original model].

As a matter of notation: remember we are using for generalized items
the same meta-variables as for ``items" (terms and abstractions).
But now the model operators will take both items and generalized items.
We shall prime the meta-variables for items (as in X', A', etc).
 *}

subsection {* Raw models  *}

record ('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs)model =
  gWls :: "'sort \<Rightarrow> 'gTerm \<Rightarrow> bool"
  gWlsAbs :: "'varSort \<times> 'sort \<Rightarrow> 'gAbs \<Rightarrow> bool"
  (*  *)
  gVar :: "'varSort \<Rightarrow> 'var \<Rightarrow> 'gTerm"
  gAbs ::
  "'varSort \<Rightarrow> 'var \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow>
   'gAbs"
  gOp ::
  "'opSym \<Rightarrow>
   ('index,('index,'bindex,'varSort,'var,'opSym)term)input \<Rightarrow> ('index,'gTerm)input \<Rightarrow>
   ('bindex,('index,'bindex,'varSort,'var,'opSym)abs)input \<Rightarrow> ('bindex,'gAbs)input \<Rightarrow>
   'gTerm"
  (*  *)
  gFresh ::
  "'varSort \<Rightarrow> 'var \<Rightarrow> ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow> bool"
  gFreshAbs ::
  "'varSort \<Rightarrow> 'var \<Rightarrow> ('index,'bindex,'varSort,'var,'opSym)abs \<Rightarrow> 'gAbs \<Rightarrow> bool"
  (*  *)
  gSwap ::
  "'varSort \<Rightarrow> 'var \<Rightarrow> 'var \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow>
   'gTerm"
  gSwapAbs ::
  "'varSort \<Rightarrow> 'var \<Rightarrow> 'var \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)abs \<Rightarrow> 'gAbs \<Rightarrow>
   'gAbs"
  (*   *)
  gSubst ::
  "'varSort \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow>
   'var \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow>
   'gTerm"
  gSubstAbs ::
  "'varSort \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm \<Rightarrow>
   'var \<Rightarrow>
   ('index,'bindex,'varSort,'var,'opSym)abs \<Rightarrow> 'gAbs \<Rightarrow>
   'gAbs"

subsection {* Well-sorted models of various kinds *}

text{*  Lifting the model operations to inputs *}

definition gFreshInp where
"gFreshInp MOD ys y inp' inp == liftAll2 (gFresh MOD ys y) inp' inp"

definition gFreshBinp where
"gFreshBinp MOD ys y binp' binp == liftAll2 (gFreshAbs MOD ys y) binp' binp"

definition gSwapInp where
"gSwapInp MOD zs z1 z2 inp' inp == lift2 (gSwap MOD zs z1 z2) inp' inp"

definition gSwapBinp where
"gSwapBinp MOD zs z1 z2 binp' binp == lift2 (gSwapAbs MOD zs z1 z2) binp' binp"

definition gSubstInp where
"gSubstInp MOD ys Y' Y y inp' inp == lift2 (gSubst MOD ys Y' Y y) inp' inp"

definition gSubstBinp where
"gSubstBinp MOD ys Y' Y y binp' binp == lift2 (gSubstAbs MOD ys Y' Y y) binp' binp"

(* *************************************************** *)
context FixSyn
begin

definition gWlsInp where
"gWlsInp MOD delta inp ==
 wlsOpS delta \<and> sameDom (arOf delta) inp \<and> liftAll2 (gWls MOD) (arOf delta) inp"

lemmas gWlsInp_defs = gWlsInp_def sameDom_def liftAll2_def

definition gWlsBinp where
"gWlsBinp MOD delta binp ==
 wlsOpS delta \<and> sameDom (barOf delta) binp \<and> liftAll2 (gWlsAbs MOD) (barOf delta) binp"

lemmas gWlsBinp_defs = gWlsBinp_def sameDom_def liftAll2_def

text{* Basic properties of the lifted model operations *}

text{* . for free inputs: *}

lemma sameDom_swapInp_gSwapInp[simp]:
assumes "wlsInp delta inp'" and "gWlsInp MOD delta inp"
shows
"sameDom
   (swapInp zs z1 z2 inp')
   (gSwapInp MOD zs z1 z2 inp' inp)"
proof-
  have "sameDom (swapInp zs z1 z2 inp') inp'"
  unfolding swapInp_def by(simp add: sameDom_refl)
  moreover have 1: "sameDom inp' inp"
  using assms unfolding wlsInp_iff gWlsInp_def
  using sameDom_sym sameDom_trans by blast
  moreover have "sameDom inp' (gSwapInp MOD zs z1 z2 inp' inp)"
  unfolding gSwapInp_def using 1 by(simp add: sameDom_refl)
  ultimately show ?thesis using sameDom_trans by blast
qed

lemma sameDom_substInp_gSubstInp[simp]:
assumes "wlsInp delta inp'" and "gWlsInp MOD delta inp"
shows
"sameDom
   (substInp ys Y' y inp')
   (gSubstInp MOD ys Y' Y y inp' inp)"
proof-
  have "sameDom (substInp ys Y' y inp') inp'"
  unfolding substInp_def2 by(simp add: sameDom_refl)
  moreover have 1: "sameDom inp' inp"
  using assms unfolding wlsInp_iff gWlsInp_def
  using sameDom_sym sameDom_trans by blast
  moreover have "sameDom inp' (gSubstInp MOD ys Y' Y y inp' inp)"
  unfolding gSubstInp_def using 1 by(simp add: sameDom_refl)
  ultimately show ?thesis using sameDom_trans by blast
qed

text{* . for bound inputs: *}

lemma sameDom_swapBinp_gSwapBinp[simp]:
assumes "wlsBinp delta binp'" and "gWlsBinp MOD delta binp"
shows
"sameDom
   (swapBinp zs z1 z2 binp')
   (gSwapBinp MOD zs z1 z2 binp' binp)"
proof-
  have "sameDom (swapBinp zs z1 z2 binp') binp'"
  unfolding swapBinp_def by(simp add: sameDom_refl)
  moreover have 1: "sameDom binp' binp"
  using assms unfolding wlsBinp_iff gWlsBinp_def
  using sameDom_sym sameDom_trans by blast
  moreover have "sameDom binp' (gSwapBinp MOD zs z1 z2 binp' binp)"
  unfolding gSwapBinp_def using 1 by(simp add: sameDom_refl)
  ultimately show ?thesis using sameDom_trans by blast
qed

lemma sameDom_substBinp_gSubstBinp[simp]:
assumes "wlsBinp delta binp'" and "gWlsBinp MOD delta binp"
shows
"sameDom
   (substBinp ys Y' y binp')
   (gSubstBinp MOD ys Y' Y y binp' binp)"
proof-
  have "sameDom (substBinp ys Y' y binp') binp'"
  unfolding substBinp_def2 by(simp add: sameDom_refl)
  moreover have 1: "sameDom binp' binp"
  using assms unfolding wlsBinp_iff gWlsBinp_def
  using sameDom_sym sameDom_trans by blast
  moreover have "sameDom binp' (gSubstBinp MOD ys Y' Y y binp' binp)"
  unfolding gSubstBinp_def using 1 by(simp add: sameDom_refl)
  ultimately show ?thesis using sameDom_trans by blast
qed

lemmas sameDom_gInput_simps =
sameDom_swapInp_gSwapInp sameDom_substInp_gSubstInp
sameDom_swapBinp_gSwapBinp sameDom_substBinp_gSubstBinp

text{* Domain disjointness: *}

definition gWlsDisj where
"gWlsDisj MOD == \<forall> s s' X. gWls MOD s X \<and> gWls MOD s' X \<longrightarrow> s = s'"

definition gWlsAbsDisj where
"gWlsAbsDisj MOD ==
 \<forall> xs s xs' s' A.
    isInBar (xs,s) \<and> isInBar (xs',s') \<and>
    gWlsAbs MOD (xs,s) A \<and> gWlsAbs MOD (xs',s') A
    \<longrightarrow> xs = xs' \<and> s = s'"

definition gWlsAllDisj where
"gWlsAllDisj MOD ==
 gWlsDisj MOD \<and> gWlsAbsDisj MOD"

lemmas gWlsAllDisj_defs =
gWlsAllDisj_def
gWlsDisj_def gWlsAbsDisj_def

text {* Abstraction domains inhabited only within bound arities: *}

definition gWlsAbsIsInBar where
"gWlsAbsIsInBar MOD ==
 \<forall> us s A. gWlsAbs MOD (us,s) A \<longrightarrow> isInBar (us,s)"

text{* Domain preservation by the operators *}

text{* The constructs preserve the domains: *}

definition gVarPresGWls where
"gVarPresGWls MOD ==
 \<forall> xs x. gWls MOD (asSort xs) (gVar MOD xs x)"

definition gAbsPresGWls where
"gAbsPresGWls MOD ==
 \<forall> xs s x X' X.
   isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   gWlsAbs MOD (xs,s) (gAbs MOD xs x X' X)"

definition gOpPresGWls where
"gOpPresGWls MOD ==
 \<forall> delta inp' inp binp' binp.
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<and> wlsBinp delta binp' \<and> gWlsBinp MOD delta binp
   \<longrightarrow> gWls MOD (stOf delta) (gOp MOD delta inp' inp binp' binp)"

definition gConsPresGWls where
"gConsPresGWls MOD ==
 gVarPresGWls MOD \<and>
 gAbsPresGWls MOD \<and>
 gOpPresGWls MOD"

lemmas gConsPresGWls_defs = gConsPresGWls_def
gVarPresGWls_def
gAbsPresGWls_def
gOpPresGWls_def

text{* ``swap" preserves the domains: *}

definition gSwapPresGWls where
"gSwapPresGWls MOD ==
 \<forall> zs z1 z2 s X' X.
   wls s X' \<and> gWls MOD s X \<longrightarrow>
   gWls MOD s (gSwap MOD zs z1 z2 X' X)"

definition gSwapAbsPresGWlsAbs where
"gSwapAbsPresGWlsAbs MOD ==
 \<forall> zs z1 z2 us s A' A.
   isInBar (us,s) \<and> wlsAbs (us,s) A' \<and> gWlsAbs MOD (us,s) A \<longrightarrow>
   gWlsAbs MOD (us,s) (gSwapAbs MOD zs z1 z2 A' A)"

definition gSwapAllPresGWlsAll where
"gSwapAllPresGWlsAll MOD ==
 gSwapPresGWls MOD \<and> gSwapAbsPresGWlsAbs MOD"

lemmas gSwapAllPresGWlsAll_defs = gSwapAllPresGWlsAll_def
gSwapPresGWls_def gSwapAbsPresGWlsAbs_def

text{* ``subst" preserves the domains: *}

definition gSubstPresGWls where
"gSubstPresGWls MOD ==
 \<forall> ys Y' Y y s X' X.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   gWls MOD s (gSubst MOD ys Y' Y y X' X)"

definition gSubstAbsPresGWlsAbs where
"gSubstAbsPresGWlsAbs MOD ==
 \<forall> ys Y' Y y us s A' A.
   isInBar (us,s) \<and>
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and> wlsAbs (us,s) A' \<and> gWlsAbs MOD (us,s) A \<longrightarrow>
   gWlsAbs MOD (us,s) (gSubstAbs MOD ys Y' Y y A' A)"

definition gSubstAllPresGWlsAll where
"gSubstAllPresGWlsAll MOD ==
 gSubstPresGWls MOD \<and> gSubstAbsPresGWlsAbs MOD"

lemmas gSubstAllPresGWlsAll_defs = gSubstAllPresGWlsAll_def
gSubstPresGWls_def gSubstAbsPresGWlsAbs_def

text{* Clauses for fresh: *}

definition gFreshGVar where
"gFreshGVar MOD ==
 \<forall> ys y xs x.
   (ys \<noteq> xs \<or> y \<noteq> x) \<longrightarrow>
   gFresh MOD ys y (Var xs x) (gVar MOD xs x)"

definition gFreshGAbs1 where
"gFreshGAbs1 MOD ==
 \<forall> ys y s X' X.
   isInBar (ys,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   gFreshAbs MOD ys y (Abs ys y X') (gAbs MOD ys y X' X)"

definition gFreshGAbs2 where
"gFreshGAbs2 MOD ==
 \<forall> ys y xs x s X' X.
   isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   fresh ys y X' \<and> gFresh MOD ys y X' X \<longrightarrow>
   gFreshAbs MOD ys y (Abs xs x X') (gAbs MOD xs x X' X)"

definition gFreshGOp where
"gFreshGOp MOD ==
 \<forall> ys y delta inp' inp binp' binp.
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<and> wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   freshInp ys y inp' \<and> gFreshInp MOD ys y inp' inp \<and>
   freshBinp ys y binp' \<and> gFreshBinp MOD ys y binp' binp \<longrightarrow>
   gFresh MOD ys y (Op delta inp' binp') (gOp MOD delta inp' inp binp' binp)"

definition gFreshCls where
"gFreshCls MOD ==
gFreshGVar MOD \<and>
gFreshGAbs1 MOD \<and> gFreshGAbs2 MOD \<and>
gFreshGOp MOD"

lemmas gFreshCls_defs = gFreshCls_def
gFreshGVar_def
gFreshGAbs1_def gFreshGAbs2_def
gFreshGOp_def

(* Clauses for swap: fully-conditional versions and less-conditional,
stronger versions (suffix ``STR") *)

definition gSwapGVar where
"gSwapGVar MOD ==
 \<forall> zs z1 z2 xs x.
   gSwap MOD zs z1 z2 (Var xs x) (gVar MOD xs x) =
   gVar MOD xs (x @xs[z1 \<and> z2]_zs)"

definition gSwapGAbs where
"gSwapGAbs MOD ==
 \<forall> zs z1 z2 xs x s X' X.
   isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   gSwapAbs MOD zs z1 z2 (Abs xs x X') (gAbs MOD xs x X' X) =
   gAbs MOD xs (x @xs[z1 \<and> z2]_zs) (X' #[z1 \<and> z2]_zs) (gSwap MOD zs z1 z2 X' X)"

definition gSwapGOp where
"gSwapGOp MOD ==
 \<forall> zs z1 z2 delta inp' inp binp' binp.
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<and> wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   gSwap MOD zs z1 z2 (Op delta inp' binp') (gOp MOD delta inp' inp binp' binp) =
   gOp MOD delta
     (inp' %[z1 \<and> z2]_zs) (gSwapInp MOD zs z1 z2 inp' inp)
     (binp' %%[z1 \<and> z2]_zs) (gSwapBinp MOD zs z1 z2 binp' binp)"

definition gSwapCls where
"gSwapCls MOD ==
gSwapGVar MOD \<and>
gSwapGAbs MOD \<and>
gSwapGOp MOD"

lemmas gSwapCls_defs = gSwapCls_def
gSwapGVar_def
gSwapGAbs_def
gSwapGOp_def

(* Clauses for subst: *)

definition gSubstGVar1 where
"gSubstGVar1 MOD ==
 \<forall> ys y Y' Y xs x.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<longrightarrow>
   (ys \<noteq> xs \<or> y \<noteq> x) \<longrightarrow>
   gSubst MOD ys Y' Y y (Var xs x) (gVar MOD xs x) =
   gVar MOD xs x"

definition gSubstGVar2 where
"gSubstGVar2 MOD ==
 \<forall> ys y Y' Y.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<longrightarrow>
   gSubst MOD ys Y' Y y (Var ys y) (gVar MOD ys y) = Y"

definition gSubstGAbs where
"gSubstGAbs MOD ==
 \<forall> ys y Y' Y xs x s X' X.
   isInBar (xs,s) \<and>
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and>
   wls s X' \<and> gWls MOD s X \<longrightarrow>
   (xs \<noteq> ys \<or> x \<noteq> y) \<and> fresh xs x Y' \<and> gFresh MOD xs x Y' Y \<longrightarrow>
   gSubstAbs MOD ys Y' Y y (Abs xs x X') (gAbs MOD xs x X' X) =
   gAbs MOD xs x (X' #[Y' / y]_ys) (gSubst MOD ys Y' Y y X' X)"

definition gSubstGOp where
"gSubstGOp MOD ==
 \<forall> ys y Y' Y delta inp' inp binp' binp.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and>
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<and>
   wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   gSubst MOD ys Y' Y y (Op delta inp' binp') (gOp MOD delta inp' inp binp' binp) =
   gOp MOD delta
     (inp' %[Y' / y]_ys) (gSubstInp MOD ys Y' Y y inp' inp)
     (binp' %%[Y' / y]_ys) (gSubstBinp MOD ys Y' Y y binp' binp)"

definition gSubstCls where
"gSubstCls MOD ==
 gSubstGVar1 MOD \<and> gSubstGVar2 MOD \<and>
 gSubstGAbs MOD \<and>
 gSubstGOp MOD"

lemmas gSubstCls_defs = gSubstCls_def
gSubstGVar1_def gSubstGVar2_def
gSubstGAbs_def
gSubstGOp_def

(* Freshness-based congruence for abstractions: *)

(* ... employing swap: *)

definition gAbsCongS where
"gAbsCongS MOD ==
 \<forall> xs x x2 y s X' X X2' X2.
   isInBar (xs,s) \<and>
   wls s X' \<and> gWls MOD s X \<and>
   wls s X2' \<and> gWls MOD s X2 \<longrightarrow>
   fresh xs y X' \<and> gFresh MOD xs y X' X \<and>
   fresh xs y X2' \<and> gFresh MOD xs y X2' X2 \<and>
   (X' #[y \<and> x]_xs) = (X2' #[y \<and> x2]_xs) \<longrightarrow>
   gSwap MOD xs y x X' X = gSwap MOD xs y x2 X2' X2 \<longrightarrow>
   gAbs MOD xs x X' X = gAbs MOD xs x2 X2' X2"

(* ... Note: no need for congruence employing subst (as it is not used in the
definition of rmorphisms *)

(* Abstraction renaming: *)

definition gAbsRen where
"gAbsRen MOD ==
 \<forall> xs y x s X' X.
    isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
    fresh xs y X' \<and> gFresh MOD xs y X' X \<longrightarrow>
    gAbs MOD xs y (X' #[y // x]_xs) (gSubst MOD xs (Var xs y) (gVar MOD xs y) x X' X) =
    gAbs MOD xs x X' X"

text {* Well-sorted fresh-swap models: *}

definition wlsFSw where
"wlsFSw MOD ==
 gWlsAllDisj MOD \<and> gWlsAbsIsInBar MOD \<and>
 gConsPresGWls MOD \<and> gSwapAllPresGWlsAll MOD \<and>
 gFreshCls MOD \<and> gSwapCls MOD \<and> gAbsCongS MOD"

lemmas wlsFSw_defs1 = wlsFSw_def
gWlsAllDisj_def gWlsAbsIsInBar_def
gConsPresGWls_def gSwapAllPresGWlsAll_def
gFreshCls_def gSwapCls_def gAbsCongS_def

lemmas wlsFSw_defs = wlsFSw_def
gWlsAllDisj_defs gWlsAbsIsInBar_def
gConsPresGWls_defs gSwapAllPresGWlsAll_defs
gFreshCls_defs gSwapCls_defs gAbsCongS_def

text {* Well-sorted fresh-subst models: *}

definition wlsFSb where
"wlsFSb MOD ==
 gWlsAllDisj MOD \<and> gWlsAbsIsInBar MOD \<and>
 gConsPresGWls MOD \<and> gSubstAllPresGWlsAll MOD \<and>
 gFreshCls MOD \<and> gSubstCls MOD \<and> gAbsRen MOD"

lemmas wlsFSb_defs1 = wlsFSb_def
gWlsAllDisj_def gWlsAbsIsInBar_def
gConsPresGWls_def gSubstAllPresGWlsAll_def
gFreshCls_def gSubstCls_def gAbsRen_def

lemmas wlsFSb_defs = wlsFSb_def
gWlsAllDisj_defs gWlsAbsIsInBar_def
gConsPresGWls_defs gSubstAllPresGWlsAll_defs
gFreshCls_defs gSubstCls_defs gAbsRen_def

text {* Well-sorted fresh-swap-subst-models *}

definition wlsFSwSb where
"wlsFSwSb MOD ==
 wlsFSw MOD \<and> gSubstAllPresGWlsAll MOD \<and> gSubstCls MOD"

lemmas wlsFSwSb_defs1 = wlsFSwSb_def
wlsFSw_def gSubstAllPresGWlsAll_def gSubstCls_def

lemmas wlsFSwSb_defs = wlsFSwSb_def
wlsFSw_def gSubstAllPresGWlsAll_defs gSubstCls_defs

text {* Well-sorted fresh-subst-swap-models *}

definition wlsFSbSw where
"wlsFSbSw MOD ==
 wlsFSb MOD \<and> gSwapAllPresGWlsAll MOD \<and> gSwapCls MOD"

lemmas wlsFSbSw_defs1 = wlsFSbSw_def
wlsFSw_def gSwapAllPresGWlsAll_def gSwapCls_def

lemmas wlsFSbSw_defs = wlsFSbSw_def
wlsFSw_def gSwapAllPresGWlsAll_defs gSwapCls_defs

text{* Extension of domain preservation (by swap and subst) to inputs: *}

text {* First for free inputs: *}

definition gSwapInpPresGWlsInp where
"gSwapInpPresGWlsInp MOD ==
 \<forall> zs z1 z2 delta inp' inp.
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<longrightarrow>
   gWlsInp MOD delta (gSwapInp MOD zs z1 z2 inp' inp)"

definition gSubstInpPresGWlsInp where
"gSubstInpPresGWlsInp MOD ==
 \<forall> ys y Y' Y delta inp' inp.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and>
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<longrightarrow>
   gWlsInp MOD delta (gSubstInp MOD ys Y' Y y inp' inp)"

lemma imp_gSwapInpPresGWlsInp:
fixes MOD :: "('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs)model"
assumes "gSwapPresGWls MOD"
shows "gSwapInpPresGWlsInp MOD"
unfolding gSwapInpPresGWlsInp_def wlsInp_iff gWlsInp_def liftAll2_def apply auto
proof-
  fix delta
  and inp' :: "('index,('index,'bindex,'varSort,'var,'opSym)term)input"
  and inp  :: "('index,'gTerm)input"
  and zs :: 'varSort and z1 z2 :: 'var
  assume 1: "sameDom (arOf delta) inp'" and 2: "sameDom (arOf delta) inp"
  hence "sameDom inp' inp" using sameDom_sym sameDom_trans by blast
  thus "sameDom (arOf delta) (gSwapInp MOD zs z1 z2 inp' inp)"
  using 1 2 unfolding gSwapInp_def by simp
next
  fix delta
  and inp' :: "('index,('index,'bindex,'varSort,'var,'opSym)term)input"
  and inp :: "('index,'gTerm)input" and zs :: 'varSort
  and z1 z2 :: 'var and i s Xsw
  let ?inpsw' = "inp' %[z1 \<and> z2]_zs"
  let ?inpsw = "gSwapInp MOD zs z1 z2 inp' inp"
  assume *:
  "\<forall>i s X'. arOf delta i = Some s \<and> inp' i = Some X' \<longrightarrow> wls s X'"
  "\<forall>i s X. arOf delta i = Some s \<and> inp i = Some X \<longrightarrow> gWls MOD s X"
  and arOf: "arOf delta i = Some s" and "?inpsw i = Some Xsw"
  then obtain X' X where
  inp': "inp' i = Some X'" and inp: "inp i = Some X"
  and Xsw: "Xsw = gSwap MOD zs z1 z2 X' X"
  unfolding gSwapInp_def lift2_def
  apply(cases "inp' i", auto) by(cases "inp i", auto)
  hence "wls s X' \<and> gWls MOD s X" using arOf * by fastforce
  thus "gWls MOD s Xsw"
  unfolding Xsw using assms
  unfolding gSwapPresGWls_def
  by simp
qed

lemma imp_gSubstInpPresGWlsInp:
fixes MOD :: "('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs)model"
assumes "gSubstPresGWls MOD"
shows "gSubstInpPresGWlsInp MOD"
unfolding gSubstInpPresGWlsInp_def wlsInp_iff gWlsInp_def liftAll2_def apply auto
proof-
  fix delta
  and inp' :: "('index,('index,'bindex,'varSort,'var,'opSym)term)input"
  and inp  :: "('index,'gTerm)input"
  and ys :: 'varSort and Y' Y y
  assume 1: "sameDom (arOf delta) inp'" and 2: "sameDom (arOf delta) inp"
  hence "sameDom inp' inp" using sameDom_sym sameDom_trans by blast
  thus "sameDom (arOf delta) (gSubstInp MOD ys Y' Y y inp' inp)"
  using 1 2 unfolding gSubstInp_def by simp
next
  fix delta
  and inp' :: "('index,('index,'bindex,'varSort,'var,'opSym)term)input"
  and inp :: "('index,'gTerm)input" and ys :: 'varSort
  and Y' Y y i s Xsb
  let ?inpsb' = "inp' %[Y' / y]_ys"
  let ?inpsb = "gSubstInp MOD ys Y' Y y inp' inp"
  assume *:
  "\<forall>i s X'. arOf delta i = Some s \<and> inp' i = Some X' \<longrightarrow> wls s X'"
  "\<forall>i s X. arOf delta i = Some s \<and> inp i = Some X \<longrightarrow> gWls MOD s X"
  and Ys: "wls (asSort ys) Y'"   "gWls MOD (asSort ys) Y"
  and arOf: "arOf delta i = Some s" and "?inpsb i = Some Xsb"
  then obtain X' X where
  inp': "inp' i = Some X'" and inp: "inp i = Some X"
  and Xsb: "Xsb = gSubst MOD ys Y' Y y X' X"
  unfolding gSubstInp_def lift2_def
  apply(cases "inp' i", auto) by(cases "inp i", auto)
  hence "wls s X' \<and> gWls MOD s X" using arOf * by fastforce
  thus "gWls MOD s Xsb"
  unfolding Xsb using assms Ys
  unfolding gSubstPresGWls_def
  by simp
qed

text {* Then for bound inputs: *}

definition gSwapBinpPresGWlsBinp where
"gSwapBinpPresGWlsBinp MOD ==
 \<forall> zs z1 z2 delta binp' binp.
   wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   gWlsBinp MOD delta (gSwapBinp MOD zs z1 z2 binp' binp)"

definition gSubstBinpPresGWlsBinp where
"gSubstBinpPresGWlsBinp MOD ==
 \<forall> ys y Y' Y delta binp' binp.
   wls (asSort ys) Y' \<and> gWls MOD (asSort ys) Y \<and>
   wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   gWlsBinp MOD delta (gSubstBinp MOD ys Y' Y y binp' binp)"

lemma imp_gSwapBinpPresGWlsBinp:
fixes MOD :: "('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs)model"
assumes "gSwapAbsPresGWlsAbs MOD"
shows "gSwapBinpPresGWlsBinp MOD"
unfolding gSwapBinpPresGWlsBinp_def wlsBinp_iff gWlsBinp_def liftAll2_def apply auto
proof-
  fix delta
  and binp' :: "('bindex,('index,'bindex,'varSort,'var,'opSym)abs)input"
  and binp  :: "('bindex,'gAbs)input"
  and zs :: 'varSort and z1 z2 :: 'var
  assume 1: "sameDom (barOf delta) binp'" and 2: "sameDom (barOf delta) binp"
  hence "sameDom binp' binp" using sameDom_sym sameDom_trans by blast
  thus "sameDom (barOf delta) (gSwapBinp MOD zs z1 z2 binp' binp)"
  using 1 2 unfolding gSwapBinp_def by simp
next
  fix delta
  and binp' :: "('bindex,('index,'bindex,'varSort,'var,'opSym)abs)input"
  and binp :: "('bindex,'gAbs)input" and zs :: 'varSort
  and z1 z2 :: 'var and i us s Asw
  let ?binpsw' = "binp' %%[z1 \<and> z2]_zs"
  let ?binpsw = "gSwapBinp MOD zs z1 z2 binp' binp"
  assume delta: "wlsOpS delta"
  and *:
  "\<forall>i us s A'. barOf delta i = Some (us,s) \<and> binp' i = Some A' \<longrightarrow> wlsAbs (us,s) A'"
  "\<forall>i us s A. barOf delta i = Some (us,s) \<and> binp i = Some A \<longrightarrow> gWlsAbs MOD (us,s) A"
  and barOf: "barOf delta i = Some (us,s)" and "?binpsw i = Some Asw"
  then obtain A' A where
  binp': "binp' i = Some A'" and binp: "binp i = Some A"
  and Asw: "Asw = gSwapAbs MOD zs z1 z2 A' A"
  unfolding gSwapBinp_def lift2_def
  apply(cases "binp' i", auto) by(cases "binp i", auto)
  hence "wlsAbs (us,s) A' \<and> gWlsAbs MOD (us,s) A" using barOf * by fastforce
  thus "gWlsAbs MOD (us,s) Asw"
  unfolding Asw using assms delta barOf
  unfolding gSwapPresGWls_def gSwapAbsPresGWlsAbs_def
  by simp
qed

lemma imp_gSubstBinpPresGWlsBinp:
fixes MOD :: "('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs)model"
assumes "gSubstAbsPresGWlsAbs MOD"
shows "gSubstBinpPresGWlsBinp MOD"
unfolding gSubstBinpPresGWlsBinp_def wlsBinp_iff gWlsBinp_def liftAll2_def apply auto
proof-
  fix delta
  and binp' :: "('bindex,('index,'bindex,'varSort,'var,'opSym)abs)input"
  and binp  :: "('bindex,'gAbs)input"
  and ys :: 'varSort and Y' Y y
  assume 1: "sameDom (barOf delta) binp'" and 2: "sameDom (barOf delta) binp"
  hence "sameDom binp' binp" using sameDom_sym sameDom_trans by blast
  thus "sameDom (barOf delta) (gSubstBinp MOD ys Y' Y y binp' binp)"
  using 1 2 unfolding gSubstBinp_def by simp
next
  fix delta
  and binp' :: "('bindex,('index,'bindex,'varSort,'var,'opSym)abs)input"
  and binp :: "('bindex,'gAbs)input" and ys :: 'varSort
  and Y' Y y i us s Asb
  let ?binpsb' = "binp' %%[Y' / y]_ys"
  let ?binpsb = "gSubstBinp MOD ys Y' Y y binp' binp"
  assume delta: "wlsOpS delta"
  and *:
  "\<forall>i us s A'. barOf delta i = Some (us,s) \<and> binp' i = Some A' \<longrightarrow> wlsAbs (us,s) A'"
  "\<forall>i us s A. barOf delta i = Some (us,s) \<and> binp i = Some A \<longrightarrow> gWlsAbs MOD (us,s) A"
  and Ys: "wls (asSort ys) Y'"   "gWls MOD (asSort ys) Y"
  and barOf: "barOf delta i = Some (us,s)" and "?binpsb i = Some Asb"
  then obtain A' A where
  binp': "binp' i = Some A'" and binp: "binp i = Some A"
  and Asb: "Asb = gSubstAbs MOD ys Y' Y y A' A"
  unfolding gSubstBinp_def lift2_def
  apply(cases "binp' i", auto) by(cases "binp i", auto)
  hence "wlsAbs (us,s) A' \<and> gWlsAbs MOD (us,s) A" using barOf * by fastforce
  thus "gWlsAbs MOD (us,s) Asb"
  unfolding Asb using assms barOf delta Ys
  unfolding gSubstAbsPresGWlsAbs_def
  by simp
qed

subsection{* Model morphisms from the term model  *}

definition presWls where
"presWls h MOD ==
 \<forall> s X. wls s X \<longrightarrow> gWls MOD s (h X)"

definition presWlsAbs where
"presWlsAbs hA MOD ==
 \<forall> us s A. wlsAbs (us,s) A \<longrightarrow> gWlsAbs MOD (us,s) (hA A)"

definition presWlsAll where
"presWlsAll h hA MOD ==
 presWls h MOD \<and> presWlsAbs hA MOD"

lemmas presWlsAll_defs = presWlsAll_def
presWls_def presWlsAbs_def

definition presVar where
"presVar h MOD ==
 \<forall> xs x. h (Var xs x) = gVar MOD xs x"

definition presAbs where
"presAbs h hA MOD ==
 \<forall> xs x s X.
   isInBar (xs,s) \<and> wls s X \<longrightarrow>
   hA (Abs xs x X) = gAbs MOD xs x X (h X)"

definition presOp where
"presOp h hA MOD ==
 \<forall> delta inp binp.
    wlsInp delta inp \<and> wlsBinp delta binp \<longrightarrow>
    h (Op delta inp binp) =
    gOp MOD delta inp (lift h inp) binp (lift hA binp)"

definition presCons where
"presCons h hA MOD ==
 presVar h MOD \<and>
 presAbs h hA MOD \<and>
 presOp h hA MOD"

lemmas presCons_defs = presCons_def
presVar_def
presAbs_def
presOp_def

definition presFresh where
"presFresh h MOD ==
 \<forall> ys y s X.
    wls s X \<longrightarrow>
    fresh ys y X \<longrightarrow> gFresh MOD ys y X (h X)"

definition presFreshAbs where
"presFreshAbs hA MOD ==
 \<forall> ys y us s A.
    wlsAbs (us,s) A \<longrightarrow>
    freshAbs ys y A \<longrightarrow> gFreshAbs MOD ys y A (hA A)"

definition presFreshAll where
"presFreshAll h hA MOD ==
 presFresh h MOD \<and> presFreshAbs hA MOD"

lemmas presFreshAll_defs = presFreshAll_def
presFresh_def presFreshAbs_def

definition presSwap where
"presSwap h MOD ==
 \<forall> zs z1 z2 s X.
    wls s X \<longrightarrow>
    h (X #[z1 \<and> z2]_zs) = gSwap MOD zs z1 z2 X (h X)"

definition presSwapAbs where
"presSwapAbs hA MOD ==
 \<forall> zs z1 z2 us s A.
    wlsAbs (us,s) A \<longrightarrow>
    hA (A $[z1 \<and> z2]_zs) = gSwapAbs MOD zs z1 z2 A (hA A)"

definition presSwapAll where
"presSwapAll h hA MOD ==
 presSwap h MOD \<and> presSwapAbs hA MOD"

lemmas presSwapAll_defs = presSwapAll_def
presSwap_def presSwapAbs_def

definition presSubst where
"presSubst h MOD ==
 \<forall> ys Y y s X.
    wls (asSort ys) Y \<and> wls s X \<longrightarrow>
    h (subst ys Y y X) = gSubst MOD ys Y (h Y) y X (h X)"

definition presSubstAbs where
"presSubstAbs h hA MOD ==
 \<forall> ys Y y us s A.
    wls (asSort ys) Y \<and> wlsAbs (us,s) A \<longrightarrow>
    hA (A $[Y / y]_ys) = gSubstAbs MOD ys Y (h Y) y A (hA A)"

definition presSubstAll where
"presSubstAll h hA MOD ==
 presSubst h MOD \<and> presSubstAbs h hA MOD"

lemmas presSubstAll_defs = presSubstAll_def
presSubst_def presSubstAbs_def

definition termFSwMorph where
"termFSwMorph h hA MOD ==
 presWlsAll h hA MOD \<and> presCons h hA MOD \<and>
 presFreshAll h hA MOD \<and> presSwapAll h hA MOD"

lemmas termFSwMorph_defs1 = termFSwMorph_def
presWlsAll_def presCons_def
presFreshAll_def presSwapAll_def

lemmas termFSwMorph_defs = termFSwMorph_def
presWlsAll_defs presCons_defs
presFreshAll_defs presSwapAll_defs

definition termFSbMorph where
"termFSbMorph h hA MOD ==
 presWlsAll h hA MOD \<and> presCons h hA MOD \<and>
 presFreshAll h hA MOD \<and> presSubstAll h hA MOD"

lemmas termFSbMorph_defs1 = termFSbMorph_def
presWlsAll_def presCons_def
presFreshAll_def presSubstAll_def

lemmas termFSbMorph_defs = termFSbMorph_def
presWlsAll_defs presCons_defs
presFreshAll_defs presSubstAll_defs

definition termFSwSbMorph where
"termFSwSbMorph h hA MOD ==
 termFSwMorph h hA MOD \<and> presSubstAll h hA MOD"

lemmas termFSwSbMorph_defs1 = termFSwSbMorph_def
termFSwMorph_def presSubstAll_def

lemmas termFSwSbMorph_defs = termFSwSbMorph_def
termFSwMorph_defs presSubstAll_defs

text{* Extension of domain preservation (by the morphisms) to inputs *}

text{* . for free inputs: *}

lemma presWls_wlsInp:
assumes *: "wlsInp delta inp"
and **: "presWls h MOD"
shows "gWlsInp MOD delta (lift h inp)"
using assms unfolding wlsInp_iff gWlsInp_def apply auto
unfolding liftAll2_def apply auto
proof-
  fix i s X
  assume delta: "arOf delta i = Some s" and "lift h inp i = Some X"
  then obtain X' where X: "X = h X'" and inp: "inp i = Some X'"
  unfolding lift_def by (cases "inp i", auto)
  hence "wls s X'" using delta * unfolding wlsInp_iff sameDom_def liftAll2_def by blast
  thus "gWls MOD s X"
  unfolding X using ** unfolding presWls_def by simp
qed

text{* . for bound inputs: *}

lemma presWls_wlsBinp:
assumes *: "wlsBinp delta binp"
and **: "presWlsAbs hA MOD"
shows "gWlsBinp MOD delta (lift hA binp)"
using assms unfolding wlsBinp_iff gWlsBinp_def apply auto
unfolding liftAll2_def apply auto
proof-
  fix i us s A
  assume delta: "barOf delta i = Some (us,s)" and "lift hA binp i = Some A"
  then obtain A' where A: "A = hA A'" and binp: "binp i = Some A'"
  unfolding lift_def by (cases "binp i", auto)
  hence "wlsAbs (us,s) A'" using delta * unfolding wlsBinp_iff sameDom_def liftAll2_def by blast
  thus "gWlsAbs MOD (us,s) A"
  unfolding A using ** unfolding presWlsAbs_def by simp
qed

subsection {* From models to iterative models *}

text {* The transition map: *}

definition fromMOD ::
"('index,'bindex,'varSort,'sort,'opSym,'var,'gTerm,'gAbs) model
 \<Rightarrow>
 ('index,'bindex,'varSort,'sort,'opSym,'var,
  ('index,'bindex,'varSort,'var,'opSym)term \<times> 'gTerm,
  ('index,'bindex,'varSort,'var,'opSym)abs \<times> 'gAbs) Iteration.model"
where
"fromMOD MOD ==
 \<lparr>
  igWls = %s X'X. wls s (fst X'X) \<and> gWls MOD s (snd X'X),
  igWlsAbs = %us_s A'A. wlsAbs us_s (fst A'A) \<and> gWlsAbs MOD us_s (snd A'A),

  igVar = %xs x. (Var xs x, gVar MOD xs x),
  igAbs = %xs x X'X. (Abs xs x (fst X'X), gAbs MOD xs x (fst X'X) (snd X'X)),
  igOp =
  %delta iinp biinp.
    (Op delta (lift fst iinp) (lift fst biinp),
     gOp MOD delta
       (lift fst iinp) (lift snd iinp)
       (lift fst biinp) (lift snd biinp)),

  igFresh =
  %ys y X'X. fresh ys y (fst X'X) \<and> gFresh MOD ys y (fst X'X) (snd X'X),
  igFreshAbs =
  %ys y A'A. freshAbs ys y (fst A'A) \<and> gFreshAbs MOD ys y (fst A'A) (snd A'A),

  igSwap =
  %zs z1 z2 X'X. ((fst X'X) #[z1 \<and> z2]_zs, gSwap MOD zs z1 z2 (fst X'X) (snd X'X)),
  igSwapAbs =
  %zs z1 z2 A'A. ((fst A'A) $[z1 \<and> z2]_zs, gSwapAbs MOD zs z1 z2 (fst A'A) (snd A'A)),

  igSubst =
  %ys Y'Y y X'X.
    ((fst X'X) #[(fst Y'Y) / y]_ys,
     gSubst MOD ys (fst Y'Y) (snd Y'Y) y (fst X'X) (snd X'X)),
  igSubstAbs =
  %ys Y'Y y A'A.
    ((fst A'A) $[(fst Y'Y) / y]_ys,
     gSubstAbs MOD ys (fst Y'Y) (snd Y'Y) y (fst A'A) (snd A'A))
 \<rparr>"

text{* Basic simplification rules: *}

lemma fromMOD_basic_simps[simp]:
"igWls (fromMOD MOD) s X'X =
 (wls s (fst X'X) \<and> gWls MOD s (snd X'X))"
(*  *)
"igWlsAbs (fromMOD MOD) us_s A'A =
 (wlsAbs us_s (fst A'A) \<and> gWlsAbs MOD us_s (snd A'A))"
(*  *)
"igVar (fromMOD MOD) xs x = (Var xs x, gVar MOD xs x)"
(*  *)
"igAbs (fromMOD MOD) xs x X'X = (Abs xs x (fst X'X), gAbs MOD xs x (fst X'X) (snd X'X))"
(*  *)
"igOp (fromMOD MOD) delta iinp biinp =
 (Op delta (lift fst iinp) (lift fst biinp),
  gOp MOD delta
    (lift fst iinp) (lift snd iinp)
    (lift fst biinp) (lift snd biinp))"
(*  *)
"igFresh (fromMOD MOD) ys y X'X =
 (fresh ys y (fst X'X) \<and> gFresh MOD ys y (fst X'X) (snd X'X))"
(*  *)
"igFreshAbs (fromMOD MOD) ys y A'A  =
 (freshAbs ys y (fst A'A) \<and> gFreshAbs MOD ys y (fst A'A) (snd A'A))"
(*  *)
"igSwap (fromMOD MOD) zs z1 z2 X'X =
 ((fst X'X) #[z1 \<and> z2]_zs, gSwap MOD zs z1 z2 (fst X'X) (snd X'X))"
(*  *)
"igSwapAbs (fromMOD MOD) zs z1 z2 A'A =
 ((fst A'A) $[z1 \<and> z2]_zs, gSwapAbs MOD zs z1 z2 (fst A'A) (snd A'A))"
(*  *)
"igSubst (fromMOD MOD) ys Y'Y y X'X =
 ((fst X'X) #[(fst Y'Y) / y]_ys,
  gSubst MOD ys (fst Y'Y) (snd Y'Y) y (fst X'X) (snd X'X))"
(*  *)
"igSubstAbs (fromMOD MOD) ys Y'Y y A'A =
 ((fst A'A) $[(fst Y'Y) / y]_ys,
  gSubstAbs MOD ys (fst Y'Y) (snd Y'Y) y (fst A'A) (snd A'A))"
unfolding fromMOD_def by auto

text{* Simps for inputs *}

text{* . for free inputs: *}

lemma igWlsInp_fromMOD[simp]:
"igWlsInp (fromMOD MOD) delta iinp =
 (wlsInp delta (lift fst iinp) \<and> gWlsInp MOD delta (lift snd iinp))"
proof-
  let ?L = "igWlsInp (fromMOD MOD) delta iinp"
  let ?R1 = "wlsInp delta (lift fst iinp)"
  let ?R2 = "gWlsInp MOD delta (lift snd iinp)"
  {assume ?L
   hence ?R1
   unfolding igWlsInp_def wlsInp_iff apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  moreover
  {assume ?L
   hence ?R2
   unfolding igWlsInp_def gWlsInp_def apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  moreover
  {assume ?R1 and ?R2
   hence ?L
   unfolding igWlsInp_def wlsInp_iff gWlsInp_def apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  ultimately show ?thesis by blast
qed

lemma igFreshInp_fromMOD[simp]:
"igFreshInp (fromMOD MOD) ys y iinp =
 (freshInp ys y (lift fst iinp) \<and>
  gFreshInp MOD ys y (lift fst iinp) (lift snd iinp))"
proof-
  let ?L = "igFreshInp (fromMOD MOD) ys y iinp"
  let ?R1 = "freshInp ys y (lift fst iinp)"
  let ?R2 = "gFreshInp MOD ys y (lift fst iinp) (lift snd iinp)"
  {assume ?L
   hence ?R1
   unfolding igFreshInp_def freshInp_def liftAll_def lift_def
   apply clarify by(case_tac "iinp i", auto)
  }
  moreover
  {assume ?L
   hence ?R2
   unfolding igFreshInp_def gFreshInp_def liftAll2_def liftAll_def lift_def
   apply clarify by(case_tac "iinp i", auto)
  }
  moreover
  {assume *: ?R1 and **: ?R2
   have ?L
   unfolding igFreshInp_def liftAll_def apply clarify
   proof-
     fix i X' X
     assume "iinp i = Some (X',X)"
     thus "igFresh (fromMOD MOD) ys y (X',X)"
     using * **
     unfolding freshInp_def gFreshInp_def liftAll2_def liftAll_def lift_def
     apply - apply(elim allE[of _ i]) by(cases "iinp i", auto)
   qed
  }
  ultimately show ?thesis by blast
qed

lemma igSwapInp_fromMOD[simp]:
"igSwapInp (fromMOD MOD) zs z1 z2 iinp =
 lift2 Pair
   (swapInp zs z1 z2 (lift fst iinp))
   (gSwapInp MOD zs z1 z2 (lift fst iinp) (lift snd iinp))"
unfolding igSwapInp_def swapInp_def gSwapInp_def lift_def lift2_def
apply(rule ext) by(case_tac "iinp i", auto)

lemma igSubstInp_fromMOD[simp]:
"igSubstInp (fromMOD MOD) ys Y'Y y iinp =
 lift2 Pair
   (substInp ys (fst Y'Y) y (lift fst iinp))
   (gSubstInp MOD ys (fst Y'Y) (snd Y'Y) y (lift fst iinp) (lift snd iinp))"
unfolding igSubstInp_def substInp_def2 gSubstInp_def lift_def lift2_def
apply(rule ext) by(case_tac "iinp i", auto)

lemmas input_fromMOD_simps =
igWlsInp_fromMOD
igFreshInp_fromMOD
igSwapInp_fromMOD
igSubstInp_fromMOD

text{* . for bound inputs: *}

lemma igWlsBinp_fromMOD[simp]:
"igWlsBinp (fromMOD MOD) delta biinp =
 (wlsBinp delta (lift fst biinp) \<and> gWlsBinp MOD delta (lift snd biinp))"
proof-
  let ?L = "igWlsBinp (fromMOD MOD) delta biinp"
  let ?R1 = "wlsBinp delta (lift fst biinp)"
  let ?R2 = "gWlsBinp MOD delta (lift snd biinp)"
  {assume ?L
   hence ?R1
   unfolding igWlsBinp_def wlsBinp_iff apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  moreover
  {assume ?L
   hence ?R2
   unfolding igWlsBinp_def gWlsBinp_def apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  moreover
  {assume *: ?R1 and **: ?R2
   hence ?L
   unfolding igWlsBinp_def wlsBinp_iff gWlsBinp_def apply simp
   unfolding liftAll2_def lift_def sameDom_def by force
  }
  ultimately show ?thesis by blast
qed

lemma igFreshBinp_fromMOD[simp]:
"igFreshBinp (fromMOD MOD) ys y biinp =
 (freshBinp ys y (lift fst biinp) \<and>
  gFreshBinp MOD ys y (lift fst biinp) (lift snd biinp))"
proof-
  let ?L = "igFreshBinp (fromMOD MOD) ys y biinp"
  let ?R1 = "freshBinp ys y (lift fst biinp)"
  let ?R2 = "gFreshBinp MOD ys y (lift fst biinp) (lift snd biinp)"
  {assume ?L
   hence ?R1
   unfolding igFreshBinp_def freshBinp_def liftAll_def lift_def
   apply clarify by(case_tac "biinp i", auto)
  }
  moreover
  {assume ?L
   hence ?R2
   unfolding igFreshBinp_def gFreshBinp_def liftAll2_def liftAll_def lift_def
   apply clarify by(case_tac "biinp i", auto)
  }
  moreover
  {assume *: ?R1 and **: ?R2
   have ?L
   unfolding igFreshBinp_def liftAll_def apply clarify
   proof-
     fix i A' A
     assume "biinp i = Some (A',A)"
     thus "igFreshAbs (fromMOD MOD) ys y (A',A)"
     using * **
     unfolding freshBinp_def gFreshBinp_def liftAll2_def liftAll_def lift_def
     apply - apply(elim allE[of _ i]) by(cases "biinp i", auto)
   qed
  }
  ultimately show ?thesis by blast
qed

lemma igSwapBinp_fromMOD[simp]:
"igSwapBinp (fromMOD MOD) zs z1 z2 biinp =
 lift2 Pair
   (swapBinp zs z1 z2 (lift fst biinp))
   (gSwapBinp MOD zs z1 z2 (lift fst biinp) (lift snd biinp))"
unfolding igSwapBinp_def swapBinp_def gSwapBinp_def lift_def lift2_def
apply(rule ext) by(case_tac "biinp i", auto)

lemma igSubstBinp_fromMOD[simp]:
"igSubstBinp (fromMOD MOD) ys Y'Y y biinp =
 lift2 Pair
   (substBinp ys (fst Y'Y) y (lift fst biinp))
   (gSubstBinp MOD ys (fst Y'Y) (snd Y'Y) y (lift fst biinp) (lift snd biinp))"
unfolding igSubstBinp_def substBinp_def2 gSubstBinp_def lift_def lift2_def
apply(rule ext) by(case_tac "biinp i", auto)

lemmas binput_fromMOD_simps =
igWlsBinp_fromMOD
igFreshBinp_fromMOD
igSwapBinp_fromMOD
igSubstBinp_fromMOD

text{* Domain disjointness: *}

lemma igWlsDisj_fromMOD[simp]:
assumes "gWlsDisj MOD"
shows "igWlsDisj (fromMOD MOD)"
using assms
unfolding igWlsDisj_def gWlsDisj_def by auto

lemma igWlsAbsDisj_fromMOD[simp]:
assumes "gWlsAbsDisj MOD"
shows "igWlsAbsDisj (fromMOD MOD)"
using assms
unfolding igWlsAbsDisj_def gWlsAbsDisj_def by fastforce

lemma igWlsAllDisj_fromMOD[simp]:
assumes "gWlsAllDisj MOD"
shows "igWlsAllDisj (fromMOD MOD)"
using assms
unfolding igWlsAllDisj_def gWlsAllDisj_def by fastforce

lemmas igWlsAllDisj_fromMOD_simps =
igWlsDisj_fromMOD
igWlsAbsDisj_fromMOD
igWlsAllDisj_fromMOD

text{* Abstractions only within IsInBar: *}

lemma igWlsAbsIsInBar_fromMOD[simp]:
assumes "gWlsAbsIsInBar MOD"
shows "igWlsAbsIsInBar (fromMOD MOD)"
using assms
unfolding gWlsAbsIsInBar_def igWlsAbsIsInBar_def by simp

text{* The constructs preserve the domains: *}

lemma igVarIPresIGWls_fromMOD[simp]:
assumes "gVarPresGWls MOD"
shows "igVarIPresIGWls (fromMOD MOD)"
using assms
unfolding igVarIPresIGWls_def gVarPresGWls_def by simp

lemma igAbsIPresIGWls_fromMOD[simp]:
assumes "gAbsPresGWls MOD"
shows "igAbsIPresIGWls (fromMOD MOD)"
using assms
unfolding igAbsIPresIGWls_def gAbsPresGWls_def by simp

lemma igOpIPresIGWls_fromMOD[simp]:
assumes "gOpPresGWls MOD"
shows "igOpIPresIGWls (fromMOD MOD)"
using assms
unfolding igOpIPresIGWls_def gOpPresGWls_def by simp

lemma igConsIPresIGWls_fromMOD[simp]:
assumes "gConsPresGWls MOD"
shows "igConsIPresIGWls (fromMOD MOD)"
using assms
unfolding igConsIPresIGWls_def gConsPresGWls_def by simp

lemmas igConsIPresIGWls_fromMOD_simps =
igVarIPresIGWls_fromMOD
igAbsIPresIGWls_fromMOD
igOpIPresIGWls_fromMOD
igConsIPresIGWls_fromMOD

text{* Swap preserves the domains: *}

lemma igSwapIPresIGWls_fromMOD[simp]:
assumes "gSwapPresGWls MOD"
shows "igSwapIPresIGWls (fromMOD MOD)"
using assms
unfolding igSwapIPresIGWls_def gSwapPresGWls_def by simp

lemma igSwapAbsIPresIGWlsAbs_fromMOD[simp]:
assumes "gSwapAbsPresGWlsAbs MOD"
shows "igSwapAbsIPresIGWlsAbs (fromMOD MOD)"
using assms
unfolding igSwapAbsIPresIGWlsAbs_def gSwapAbsPresGWlsAbs_def by simp

lemma igSwapAllIPresIGWlsAll_fromMOD[simp]:
assumes "gSwapAllPresGWlsAll MOD"
shows "igSwapAllIPresIGWlsAll (fromMOD MOD)"
using assms
unfolding igSwapAllIPresIGWlsAll_def gSwapAllPresGWlsAll_def by simp

lemmas igSwapAllIPresIGWlsAll_fromMOD_simps =
igSwapIPresIGWls_fromMOD
igSwapAbsIPresIGWlsAbs_fromMOD
igSwapAllIPresIGWlsAll_fromMOD

text{* Subst preserves the domains: *}

lemma igSubstIPresIGWls_fromMOD[simp]:
assumes "gSubstPresGWls MOD"
shows "igSubstIPresIGWls (fromMOD MOD)"
using assms
unfolding igSubstIPresIGWls_def gSubstPresGWls_def by simp

lemma igSubstAbsIPresIGWlsAbs_fromMOD[simp]:
assumes "gSubstAbsPresGWlsAbs MOD"
shows "igSubstAbsIPresIGWlsAbs (fromMOD MOD)"
using assms
unfolding igSubstAbsIPresIGWlsAbs_def gSubstAbsPresGWlsAbs_def by simp

lemma igSubstAllIPresIGWlsAll_fromMOD[simp]:
assumes "gSubstAllPresGWlsAll MOD"
shows "igSubstAllIPresIGWlsAll (fromMOD MOD)"
using assms
unfolding igSubstAllIPresIGWlsAll_def gSubstAllPresGWlsAll_def by simp

lemmas igSubstAllIPresIGWlsAll_fromMOD_simps =
igSubstIPresIGWls_fromMOD
igSubstAbsIPresIGWlsAbs_fromMOD
igSubstAllIPresIGWlsAll_fromMOD

text{* The fresh clauses: *}

lemma igFreshIGVar_fromMOD[simp]:
assumes "gFreshGVar MOD"
shows "igFreshIGVar (fromMOD MOD)"
using assms
unfolding igFreshIGVar_def gFreshGVar_def by simp

lemma igFreshIGAbs1_fromMOD[simp]:
assumes "gFreshGAbs1 MOD"
shows "igFreshIGAbs1 (fromMOD MOD)"
using assms
unfolding igFreshIGAbs1_def gFreshGAbs1_def by auto

lemma igFreshIGAbs2_fromMOD[simp]:
assumes "gFreshGAbs2 MOD"
shows "igFreshIGAbs2 (fromMOD MOD)"
using assms
unfolding igFreshIGAbs2_def gFreshGAbs2_def by auto

lemma igFreshIGOp_fromMOD[simp]:
assumes "gFreshGOp MOD"
shows "igFreshIGOp (fromMOD MOD)"
using assms
unfolding igFreshIGOp_def gFreshGOp_def by simp

lemma igFreshCls_fromMOD[simp]:
assumes "gFreshCls MOD"
shows "igFreshCls (fromMOD MOD)"
using assms
unfolding igFreshCls_def gFreshCls_def by simp

lemmas igFreshCls_fromMOD_simps =
igFreshIGVar_fromMOD
igFreshIGAbs1_fromMOD igFreshIGAbs2_fromMOD
igFreshIGOp_fromMOD
igFreshCls_fromMOD

text{* The swap clauses *}

lemma igSwapIGVar_fromMOD[simp]:
assumes "gSwapGVar MOD"
shows "igSwapIGVar (fromMOD MOD)"
using assms
unfolding igSwapIGVar_def gSwapGVar_def by simp

lemma igSwapIGAbs_fromMOD[simp]:
assumes "gSwapGAbs MOD"
shows "igSwapIGAbs (fromMOD MOD)"
using assms
unfolding igSwapIGAbs_def gSwapGAbs_def by auto

lemma igSwapIGOp_fromMOD[simp]:
assumes "gSwapGOp MOD"
shows "igSwapIGOp (fromMOD MOD)"
using assms
unfolding igSwapIGOp_def gSwapGOp_def apply simp
unfolding lift_lift2 by auto

lemma igSwapCls_fromMOD[simp]:
assumes "gSwapCls MOD"
shows "igSwapCls (fromMOD MOD)"
using assms
unfolding igSwapCls_def gSwapCls_def by simp

lemmas igSwapCls_fromMOD_simps =
igSwapIGVar_fromMOD
igSwapIGAbs_fromMOD
igSwapIGOp_fromMOD
igSwapCls_fromMOD

text{* The subst clauses *}

lemma igSubstIGVar1_fromMOD[simp]:
assumes "gSubstGVar1 MOD"
shows "igSubstIGVar1 (fromMOD MOD)"
using assms
unfolding igSubstIGVar1_def gSubstGVar1_def by simp

lemma igSubstIGVar2_fromMOD[simp]:
assumes "gSubstGVar2 MOD"
shows "igSubstIGVar2 (fromMOD MOD)"
using assms
unfolding igSubstIGVar2_def gSubstGVar2_def by simp

lemma igSubstIGAbs_fromMOD[simp]:
assumes "gSubstGAbs MOD"
shows "igSubstIGAbs (fromMOD MOD)"
using assms
unfolding igSubstIGAbs_def gSubstGAbs_def by fastforce+

lemma igSubstIGOp_fromMOD[simp]:
assumes "gSubstGOp MOD"
shows "igSubstIGOp (fromMOD MOD)"
using assms
unfolding igSubstIGOp_def gSubstGOp_def apply simp
unfolding lift_lift2 by auto

lemma igSubstCls_fromMOD[simp]:
assumes "gSubstCls MOD"
shows "igSubstCls (fromMOD MOD)"
using assms
unfolding igSubstCls_def gSubstCls_def by simp

lemmas igSubstCls_fromMOD_simps =
igSubstIGVar1_fromMOD igSubstIGVar2_fromMOD
igSubstIGAbs_fromMOD
igSubstIGOp_fromMOD
igSubstCls_fromMOD

text{* Abstraction swapping congruence: *}

lemma igAbsCongS_fromMOD[simp]:
assumes "gAbsCongS MOD"
shows "igAbsCongS (fromMOD MOD)"
using assms
unfolding igAbsCongS_def gAbsCongS_def apply simp apply clarify
apply(intro conjI)
apply(erule wls_Abs_swap_cong) by blast+

text{* Abstraction renaming: *}

lemma igAbsRen_fromMOD[simp]:
assumes "gAbsRen MOD"
shows "igAbsRen (fromMOD MOD)"
using assms
unfolding igAbsRen_def gAbsRen_def vsubst_def by auto

text{* Models: *}

lemma iwlsFSw_fromMOD[simp]:
assumes "wlsFSw MOD"
shows "iwlsFSw (fromMOD MOD)"
using assms
unfolding iwlsFSw_def wlsFSw_def by simp

lemma iwlsFSb_fromMOD[simp]:
assumes "wlsFSb MOD"
shows "iwlsFSb (fromMOD MOD)"
using assms
unfolding iwlsFSb_def wlsFSb_def by simp

lemma iwlsFSwSb_fromMOD[simp]:
assumes "wlsFSwSb MOD"
shows "iwlsFSwSb (fromMOD MOD)"
using assms
unfolding iwlsFSwSb_def wlsFSwSb_def by simp

lemma iwlsFSbSw_fromMOD[simp]:
assumes "wlsFSbSw MOD"
shows "iwlsFSbSw (fromMOD MOD)"
using assms
unfolding iwlsFSbSw_def wlsFSbSw_def by simp

lemmas iwlsModel_fromMOD_simps =
iwlsFSw_fromMOD iwlsFSb_fromMOD
iwlsFSwSb_fromMOD iwlsFSbSw_fromMOD

(******************************)
lemmas fromMOD_predicate_simps =
igWlsAllDisj_fromMOD_simps
igConsIPresIGWls_fromMOD_simps
igSwapAllIPresIGWlsAll_fromMOD_simps
igSubstAllIPresIGWlsAll_fromMOD_simps
igFreshCls_fromMOD_simps
igSwapCls_fromMOD_simps
igSubstCls_fromMOD_simps
igAbsCongS_fromMOD
igAbsRen_fromMOD
iwlsModel_fromMOD_simps

lemmas fromMOD_simps =
fromMOD_basic_simps
input_fromMOD_simps
binput_fromMOD_simps
fromMOD_predicate_simps

subsection {* The recursion-iteration ``identity trick"  *}

text {* Here we show that any construct-preserving map from terms to ``fromMOD MOD"
is the identity on its first projection -- this is the main trick when
reducing recursion to iteration.  *}

lemma ipresCons_fromMOD_fst:
assumes "ipresCons h hA (fromMOD MOD)"
shows
"(wls s X \<longrightarrow> fst (h X) = X)
 \<and>
 (wlsAbs (us,s') A \<longrightarrow> fst (hA A) = A)"
apply(rule wls_rawInduct)
using assms unfolding ipresCons_def ipresVar_def apply simp
proof-
  fix delta inp binp
  assume inp: "wlsInp delta inp"  and binp: "wlsBinp delta binp"
  and IH: "liftAll2 (\<lambda>a b. fst (h b) = b) (arOf delta) inp"
  and IH': "liftAll2 (\<lambda>a b. fst (hA b) = b) (barOf delta) binp"
  have "lift (fst \<circ> h) inp = inp"
  unfolding lift_def apply(rule ext, case_tac "inp i", simp_all)
  proof-
    fix i X assume inpi: "inp i = Some X"
    then obtain s where arOfi: "arOf delta i = Some s"
    using inp unfolding wlsInp_iff sameDom_def by(cases "arOf delta i", auto)
    hence "wls s X" using inp inpi
    unfolding wlsInp_iff liftAll2_def by(cases "arOf delta i", auto)
    thus "fst (h X) = X"
    using IH inpi arOfi unfolding liftAll2_def by auto
  qed
  moreover have "lift (fst \<circ> hA) binp = binp"
  unfolding lift_def apply(rule ext, case_tac "binp i", simp_all)
  proof-
    fix i A assume binpi: "binp i = Some A"
    then obtain us s where barOfi: "barOf delta i = Some (us,s)"
    using binp unfolding wlsBinp_iff sameDom_def by(cases "barOf delta i", auto)
    hence "wlsAbs (us,s) A" using binp binpi
    unfolding wlsBinp_iff liftAll2_def by(cases "barOf delta i", auto)
    thus "fst (hA A) = A"
    using IH' binpi barOfi unfolding liftAll2_def by fastforce
  qed
  ultimately show "fst (h (Op delta inp binp)) = Op delta inp binp"
  using assms inp binp unfolding ipresCons_def ipresOp_def
  apply clarify apply (elim allE[of _ delta] allE[of _ inp] allE[of _ binp])
  apply simp unfolding lift_comp by simp
next
  fix s xs x X
  assume "isInBar (xs, s)" and "wls s X" and "fst (h X) = X"
  thus "fst (hA (Abs xs x X)) = Abs xs x X"
  using assms unfolding ipresCons_def ipresAbs_def
  apply clarify apply (elim allE[of _ xs] allE[of _ x])
  by fastforce
qed

lemma ipresCons_fromMOD_fst_simps[simp]:
"\<lbrakk>ipresCons h hA (fromMOD MOD); wls s X\<rbrakk>
 \<Longrightarrow> fst (h X) = X"
(*  *)
"\<lbrakk>ipresCons h hA (fromMOD MOD); wlsAbs (us,s') A\<rbrakk>
 \<Longrightarrow> fst (hA A) = A"
using ipresCons_fromMOD_fst by blast+

lemma ipresCons_fromMOD_fst_inp[simp]:
assumes *: "ipresCons h hA (fromMOD MOD)"
and "wlsInp delta inp"
shows "lift (fst o h) inp = inp"
unfolding lift_def apply(rule ext)
using assms apply(case_tac "inp i", simp_all)
proof-
  fix i X assume inpi: "inp i = Some X" and inp: "wlsInp delta inp"
  then obtain s where arOfi: "arOf delta i = Some s"
  using inp unfolding wlsInp_iff sameDom_def by(cases "arOf delta i", auto)
  hence "wls s X" using inp inpi
  unfolding wlsInp_iff liftAll2_def by(cases "arOf delta i", auto)
  thus "fst (h X) = X" using * by simp
qed

lemma ipresCons_fromMOD_fst_binp[simp]:
assumes *: "ipresCons h hA (fromMOD MOD)"
and "wlsBinp delta binp"
shows
"lift (fst o hA) binp = binp"
unfolding lift_def apply(rule ext)
using assms apply(case_tac "binp i", simp_all)
proof-
  fix i A assume binpi: "binp i = Some A" and binp: "wlsBinp delta binp"
  then obtain us s where barOfi: "barOf delta i = Some (us,s)"
  using binp unfolding wlsBinp_iff sameDom_def by(cases "barOf delta i", auto)
  hence "wlsAbs (us,s) A" using binp binpi
  unfolding wlsBinp_iff liftAll2_def by(cases "barOf delta i", auto)
  thus "fst (hA A) = A" using * by simp
qed

lemmas ipresCons_fromMOD_fst_all_simps =
ipresCons_fromMOD_fst_simps
ipresCons_fromMOD_fst_inp
ipresCons_fromMOD_fst_binp

subsection {* From iteration morphisms to morphisms *}

text{* The transition map: *}

definition fromIMor ::
"(('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow>
  ('index,'bindex,'varSort,'var,'opSym)term \<times> 'gTerm)
 \<Rightarrow>
 (('index,'bindex,'varSort,'var,'opSym)term \<Rightarrow> 'gTerm)"
where
"fromIMor h == snd o h"

definition fromIMorAbs ::
"(('index,'bindex,'varSort,'var,'opSym)abs \<Rightarrow>
  ('index,'bindex,'varSort,'var,'opSym)abs \<times> 'gAbs)
 \<Rightarrow>
 (('index,'bindex,'varSort,'var,'opSym)abs \<Rightarrow> 'gAbs)"
where
"fromIMorAbs hA == snd o hA"

text{* Basic simplification rules: *}

lemma fromIMor[simp]:
"fromIMor h X' = snd (h X')"
unfolding fromIMor_def by simp

lemma fromIMorAbs[simp]:
"fromIMorAbs hA A' = snd (hA A')"
unfolding fromIMorAbs_def by simp

lemma fromIMor_snd_inp[simp]:
assumes "wlsInp delta inp"
shows
"lift (fromIMor h) inp = lift (snd o h) inp"
unfolding lift_def apply(rule ext)
by(case_tac "inp i", auto)

lemma fromIMorAbs_snd_binp[simp]:
assumes "wlsBinp delta binp"
shows
"lift (fromIMorAbs hA) binp = lift (snd o hA) binp"
unfolding lift_def apply(rule ext)
by(case_tac "binp i", auto)

lemmas fromIMor_basic_simps =
fromIMor fromIMorAbs
fromIMor_snd_inp fromIMorAbs_snd_binp

text{* Predicate simplification rules *}

text{* Domain preservation *}

lemma presWls_fromIMor[simp]:
assumes "ipresWls h (fromMOD MOD)"
shows "presWls (fromIMor h) MOD"
using assms
unfolding ipresWls_def presWls_def by simp

lemma presWlsAbs_fromIMorAbs[simp]:
assumes "ipresWlsAbs hA (fromMOD MOD)"
shows "presWlsAbs (fromIMorAbs hA) MOD"
using assms
unfolding ipresWlsAbs_def presWlsAbs_def by simp

lemma presWlsAll_fromIMorAll[simp]:
assumes "ipresWlsAll h hA (fromMOD MOD)"
shows "presWlsAll (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresWlsAll_def presWlsAll_def by simp

lemmas presWlsAll_fromIMorAll_simps =
presWls_fromIMor
presWlsAbs_fromIMorAbs
presWlsAll_fromIMorAll

text{* Preservation of the constructs *}

lemma presVar_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
shows "presVar (fromIMor h) MOD"
using assms
unfolding ipresCons_def ipresVar_def presVar_def by simp

lemma presAbs_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
shows "presAbs (fromIMor h) (fromIMorAbs hA) MOD"
using assms unfolding ipresCons_def ipresAbs_def presAbs_def
using assms apply simp by force

lemma presOp_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
shows "presOp (fromIMor h) (fromIMorAbs hA) MOD"
using assms unfolding ipresCons_def ipresOp_def presOp_def
using assms apply simp unfolding lift_comp by auto

lemma presCons_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
shows "presCons (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresCons_def presCons_def
using assms by simp

lemmas presCons_fromIMor_simps =
presVar_fromIMor
presAbs_fromIMor
presOp_fromIMor
presCons_fromIMor

text{* Preservation of freshness *}

lemma presFresh_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresFresh h (fromMOD MOD)"
shows "presFresh (fromIMor h) MOD"
using assms
unfolding ipresFresh_def presFresh_def by simp

lemma presFreshAbs_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresFreshAbs hA (fromMOD MOD)"
shows "presFreshAbs (fromIMorAbs hA) MOD"
using assms
unfolding ipresFreshAbs_def presFreshAbs_def by simp

lemma presFreshAll_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresFreshAll h hA (fromMOD MOD)"
shows "presFreshAll (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresFreshAll_def presFreshAll_def by simp

lemmas presFreshAll_fromIMor_simps =
presFresh_fromIMor
presFreshAbs_fromIMor
presFreshAll_fromIMor

text{* Preservation of swap *}

lemma presSwap_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSwap h (fromMOD MOD)"
shows "presSwap (fromIMor h) MOD"
using assms
unfolding ipresSwap_def presSwap_def by simp

lemma presSwapAbs_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSwapAbs hA (fromMOD MOD)"
shows "presSwapAbs (fromIMorAbs hA) MOD"
using assms
unfolding ipresSwapAbs_def presSwapAbs_def by simp

lemma presSwapAll_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSwapAll h hA (fromMOD MOD)"
shows "presSwapAll (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresSwapAll_def presSwapAll_def by simp

lemmas presSwapAll_fromIMor_simps =
presSwap_fromIMor
presSwapAbs_fromIMor
presSwapAll_fromIMor

text{* Preservation of subst *}

lemma presSubst_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSubst h (fromMOD MOD)"
shows "presSubst (fromIMor h) MOD"
using assms
unfolding ipresSubst_def presSubst_def by auto

lemma presSubstAbs_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSubstAbs h hA (fromMOD MOD)"
shows "presSubstAbs (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresSubstAbs_def presSubstAbs_def by auto

lemma presSubstAll_fromIMor[simp]:
assumes "ipresCons h hA (fromMOD MOD)"
and "ipresSubstAll h hA (fromMOD MOD)"
shows "presSubstAll (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding ipresSubstAll_def presSubstAll_def by simp

lemmas presSubstAll_fromIMor_simps =
presSubst_fromIMor
presSubstAbs_fromIMor
presSubstAll_fromIMor

text{* Morphisms *}

lemma fromIMor_termFSwMorph[simp]:
assumes "termFSwImorph h hA (fromMOD MOD)"
shows "termFSwMorph (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding termFSwImorph_def termFSwMorph_def by simp

lemma fromIMor_termFSbMorph[simp]:
assumes "termFSbImorph h hA (fromMOD MOD)"
shows "termFSbMorph (fromIMor h) (fromIMorAbs hA) MOD"
using assms
unfolding termFSbImorph_def termFSbMorph_def by simp

lemma fromIMor_termFSwSbMorph[simp]:
assumes "termFSwSbImorph h hA (fromMOD MOD)"
shows "termFSwSbMorph (fromIMor h) (fromIMorAbs hA) MOD"
using assms unfolding termFSwSbImorph_defs1
using assms unfolding termFSwSbImorph_def termFSwSbMorph_def by simp

lemmas mor_fromIMor_simps =
fromIMor_termFSwMorph
fromIMor_termFSbMorph
fromIMor_termFSwSbMorph

(********************************)
lemmas fromIMor_predicate_simps =
presCons_fromIMor_simps
presFreshAll_fromIMor_simps
presSwapAll_fromIMor_simps
presSubstAll_fromIMor_simps
mor_fromIMor_simps

lemmas fromIMor_simps =
fromIMor_basic_simps
fromIMor_predicate_simps

subsection {* The recursion theorem *}

text{* The recursion maps: *}

definition rec where
"rec MOD == fromIMor (iter (fromMOD MOD))"

definition recAbs where
"recAbs MOD == fromIMorAbs (iterAbs (fromMOD MOD))"

text{* Existence: *}

theorem wlsFSw_recAll_termFSwMorph:
assumes "wlsFSw MOD"
shows "termFSwMorph (rec MOD) (recAbs MOD) MOD"
proof-
  let ?IMOD = "fromMOD MOD"
  have "iwlsFSw ?IMOD" using assms by simp
  hence "termFSwImorph (iter ?IMOD) (iterAbs ?IMOD) ?IMOD"
  by(rule iwlsFSw_iterAll_termFSwImorph)
  thus ?thesis unfolding rec_def recAbs_def by simp
qed

theorem wlsFSb_recAll_termFSbMorph:
assumes "wlsFSb MOD"
shows "termFSbMorph (rec MOD) (recAbs MOD) MOD"
proof-
  let ?IMOD = "fromMOD MOD"
  have "iwlsFSb ?IMOD" using assms by simp
  hence "termFSbImorph (iter ?IMOD) (iterAbs ?IMOD) ?IMOD"
  by(rule iwlsFSb_iterAll_termFSbImorph)
  thus ?thesis unfolding rec_def recAbs_def by simp
qed

theorem wlsFSwSb_recAll_termFSwSbMorph:
assumes "wlsFSwSb MOD"
shows "termFSwSbMorph (rec MOD) (recAbs MOD) MOD"
proof-
  let ?IMOD = "fromMOD MOD"
  have "iwlsFSwSb ?IMOD" using assms by simp
  hence "termFSwSbImorph (iter ?IMOD) (iterAbs ?IMOD) ?IMOD"
  by(rule iwlsFSwSb_iterAll_termFSwSbImorph)
  thus ?thesis unfolding rec_def recAbs_def by simp
qed

theorem wlsFSbSw_recAll_termFSwSbMorph:
assumes "wlsFSbSw MOD"
shows "termFSwSbMorph (rec MOD) (recAbs MOD) MOD"
proof-
  let ?IMOD = "fromMOD MOD"
  have "iwlsFSbSw ?IMOD" using assms by simp
  hence "termFSwSbImorph (iter ?IMOD) (iterAbs ?IMOD) ?IMOD"
  by(rule iwlsFSbSw_iterAll_termFSwSbImorph)
  thus ?thesis unfolding rec_def recAbs_def by simp
qed

text{* Uniqueness: *}

lemma presCons_unique:
assumes "presCons f fA MOD" and "presCons g gA MOD"
shows
"(wls s X \<longrightarrow> f X = g X) \<and>
 (wlsAbs (us,s') A \<longrightarrow> fA A = gA A)"
proof(rule wls_rawInduct)
  fix xs x
  show "f (Var xs x) = g (Var xs x)"
  using assms unfolding presCons_def presVar_def by simp
next
  fix delta inp binp
  assume inp_wls: "wlsInp delta inp"  "wlsBinp delta binp"
  and IH: "liftAll2 (\<lambda>s X. f X = g X) (arOf delta) inp"
          "liftAll2 (\<lambda>us_s A. fA A = gA A) (barOf delta) binp"
  have "lift f inp = lift g inp"
  proof(rule ext)
    fix i
    show "lift f inp i = lift g inp i"
    proof(cases "inp i", simp add: lift_def)
      fix X assume inp: "inp i = Some X"
      then obtain s where "arOf delta i = Some s"
      using inp_wls unfolding wlsInp_iff sameDom_def by fastforce
      thus ?thesis using inp IH unfolding liftAll2_def lift_def by auto
    qed
  qed
  moreover
  have "lift fA binp = lift gA binp"
  proof(rule ext)
    fix i
    show "lift fA binp i = lift gA binp i"
    proof(cases "binp i", simp add: lift_def)
      fix A assume binp: "binp i = Some A"
      then obtain us s where "barOf delta i = Some (us,s)"
      using inp_wls unfolding wlsBinp_iff sameDom_def by force
      thus ?thesis using binp IH unfolding liftAll2_def lift_def
      by auto
    qed
  qed
  ultimately
  show "f (Op delta inp binp) = g (Op delta inp binp)"
  using assms inp_wls
  unfolding presCons_def presOp_def by simp
next
  fix s xs x X assume "isInBar (xs, s)" and "wls s X" and "f X = g X"
  thus "fA (Abs xs x X) = gA (Abs xs x X)"
  using assms unfolding presCons_def presAbs_def apply clarify
  apply(erule allE[of _ xs], erule allE[of _ x])+ by fastforce
qed

theorem wlsFSw_recAll_unique_presCons:
assumes *: "wlsFSw MOD" and **: "presCons h hA MOD"
shows
"(wls s X \<longrightarrow> h X = rec MOD X) \<and>
 (wlsAbs (us,s') A \<longrightarrow> hA A = recAbs MOD A)"
apply(rule presCons_unique[of _ _ MOD])
using ** apply simp
using * wlsFSw_recAll_termFSwMorph unfolding termFSwMorph_def by auto

theorem wlsFSb_recAll_unique_presCons:
assumes *: "wlsFSb MOD" and **: "presCons h hA MOD"
shows
"(wls s X \<longrightarrow> h X = rec MOD X) \<and>
 (wlsAbs (us,s') A \<longrightarrow> hA A = recAbs MOD A)"
apply(rule presCons_unique[of _ _ MOD])
using ** apply simp
using * wlsFSb_recAll_termFSbMorph unfolding termFSbMorph_def by auto

theorem wlsFSwSb_recAll_unique_presCons:
assumes *: "wlsFSwSb MOD" and **: "presCons h hA MOD"
shows
"(wls s X \<longrightarrow> h X = rec MOD X) \<and>
 (wlsAbs (us,s') A \<longrightarrow> hA A = recAbs MOD A)"
using assms unfolding wlsFSwSb_def
using wlsFSw_recAll_unique_presCons by blast

theorem wlsFSbSw_recAll_unique_presCons:
assumes *: "wlsFSbSw MOD" and **: "presCons h hA MOD"
shows
"(wls s X \<longrightarrow> h X = rec MOD X) \<and>
 (wlsAbs (us,s') A \<longrightarrow> hA A = recAbs MOD A)"
using assms unfolding wlsFSbSw_def
using wlsFSb_recAll_unique_presCons by blast

subsection{* Models ``closer" to the term model *}

text{* We describe various conditions (later referred to as ``extra clauses"
or ``extra conditions")
that, when satisfied by models,
yield the recursive maps
(1) freshness-preserving and/or (2) injective and/or (3) surjective, thus bringing the
considered models ``closer" to (being isomorphic to) the term model.
The extreme case, when all of (1)-(3) above are ensured, means indeed isomorphism to
the term model -- this is in fact an abstract characterization of the term model.  *}

subsubsection {* Relevant predicates on models *}

text{* The fresh clauses reversed *}

definition gFreshGVarRev where
"gFreshGVarRev MOD ==
 \<forall> xs y x.
   gFresh MOD xs y (Var xs x) (gVar MOD xs x) \<longrightarrow>
   y \<noteq> x"

definition gFreshGAbsRev where
"gFreshGAbsRev MOD ==
 \<forall> ys y xs x s X' X.
   isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
   gFreshAbs MOD ys y (Abs xs x X') (gAbs MOD xs x X' X) \<longrightarrow>
   (ys = xs \<and> y = x) \<or> gFresh MOD ys y X' X"

definition gFreshGOpRev where
"gFreshGOpRev MOD ==
 \<forall> ys y delta inp' inp binp' binp.
   wlsInp delta inp' \<and> gWlsInp MOD delta inp \<and> wlsBinp delta binp' \<and> gWlsBinp MOD delta binp \<longrightarrow>
   gFresh MOD ys y (Op delta inp' binp') (gOp MOD delta inp' inp binp' binp) \<longrightarrow>
   gFreshInp MOD ys y inp' inp \<and> gFreshBinp MOD ys y binp' binp"

definition gFreshClsRev where
"gFreshClsRev MOD ==
gFreshGVarRev MOD \<and>
gFreshGAbsRev MOD \<and>
gFreshGOpRev MOD"

lemmas gFreshClsRev_defs = gFreshClsRev_def
gFreshGVarRev_def
gFreshGAbsRev_def
gFreshGOpRev_def

text{* Injectiveness of the construct operators *}

definition gVarInj where
"gVarInj MOD ==
 \<forall> xs x y. gVar MOD xs x = gVar MOD xs y \<longrightarrow> x = y"

definition gAbsInj where
"gAbsInj MOD ==
 \<forall> xs s x X' X X1' X1.
   isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<and> wls s X1' \<and> gWls MOD s X1 \<and>
   gAbs MOD xs x X' X = gAbs MOD xs x X1' X1
   \<longrightarrow>
   X = X1"

definition gOpInj where
"gOpInj MOD ==
 \<forall> delta delta1 inp' binp' inp binp inp1' binp1' inp1 binp1.
   wlsInp delta inp' \<and> wlsBinp delta binp' \<and> gWlsInp MOD delta inp \<and> gWlsBinp MOD delta binp \<and>
   wlsInp delta1 inp1' \<and> wlsBinp delta1 binp1' \<and> gWlsInp MOD delta1 inp1 \<and> gWlsBinp MOD delta1 binp1 \<and>
   stOf delta = stOf delta1 \<and>
   gOp MOD delta inp' inp binp' binp = gOp MOD delta1 inp1' inp1 binp1' binp1
   \<longrightarrow>
   delta = delta1 \<and> inp = inp1 \<and> binp = binp1"

definition gVarGOpInj where
"gVarGOpInj MOD ==
 \<forall> xs x delta inp' binp' inp binp.
   wlsInp delta inp' \<and> wlsBinp delta binp' \<and> gWlsInp MOD delta inp \<and> gWlsBinp MOD delta binp \<and>
   asSort xs = stOf delta
   \<longrightarrow>
   gVar MOD xs x \<noteq> gOp MOD delta inp' inp binp' binp"

definition gConsInj where
"gConsInj MOD ==
 gVarInj MOD \<and>
 gAbsInj MOD \<and>
 gOpInj MOD \<and>
 gVarGOpInj MOD"

lemmas gConsInj_defs = gConsInj_def
gVarInj_def
gAbsInj_def
gOpInj_def
gVarGOpInj_def

text{* Abstraction renaming for swapping *}

definition gAbsRenS where
"gAbsRenS MOD ==
 \<forall> xs y x s X' X.
    isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<longrightarrow>
    fresh xs y X' \<and> gFresh MOD xs y X' X \<longrightarrow>
    gAbs MOD xs y (X' #[y \<and> x]_xs) (gSwap MOD xs y x X' X) =
    gAbs MOD xs x X' X"

text{* Indifference to the general-recursive argument *}

text{* . This ``indifference" property says that the construct operators
from the model only depend on
the generalized item (i.e., generalized term or abstraction) argument,
and {\em not} on the ``item" (i.e., concrete term or abstraction) argument.
In other words, the model constructs correspond to {\em iterative clauses},
and not to the more general notion of ``general-recursive" clause.  *}

definition gAbsIndif where
"gAbsIndif MOD ==
 \<forall> xs s x X1' X2' X.
    isInBar (xs,s) \<and> wls s X1' \<and> wls s X2' \<and> gWls MOD s X \<longrightarrow>
    gAbs MOD xs x X1' X = gAbs MOD xs x X2' X"

definition gOpIndif where
"gOpIndif MOD ==
 \<forall> delta inp1' inp2' inp binp1' binp2' binp.
   wlsInp delta inp1' \<and> wlsBinp delta binp1' \<and> wlsInp delta inp2' \<and> wlsBinp delta binp2' \<and>
   gWlsInp MOD delta inp \<and> gWlsBinp MOD delta binp
   \<longrightarrow>
   gOp MOD delta inp1' inp binp1' binp = gOp MOD delta inp2' inp binp2' binp"

definition gConsIndif where
"gConsIndif MOD ==
 gOpIndif MOD \<and> gAbsIndif MOD"

lemmas gConsIndif_defs = gConsIndif_def
gAbsIndif_def gOpIndif_def

text{* Inductiveness *}

text{* . Inductiveness of a model means the satisfaction of a minimal inductive
principle (``minimal" in the sense that no fancy swapping or freshness
induction-friendly conditions are involved).     *}

definition gInduct where
"gInduct MOD ==
 \<forall> phi phiAbs s X us s' A.
   (
    (\<forall> xs x. phi (asSort xs) (gVar MOD xs x))
    \<and>
    (\<forall> delta inp' inp binp' binp.
       wlsInp delta inp' \<and> wlsBinp delta binp' \<and> gWlsInp MOD delta inp \<and> gWlsBinp MOD delta binp \<and>
       liftAll2 phi (arOf delta) inp \<and> liftAll2 phiAbs (barOf delta) binp
       \<longrightarrow> phi (stOf delta) (gOp MOD delta inp' inp binp' binp))
    \<and>
    (\<forall> xs s x X' X.
        isInBar (xs,s) \<and> wls s X' \<and> gWls MOD s X \<and>
        phi s X
        \<longrightarrow> phiAbs (xs,s) (gAbs MOD xs x X' X))
   )
   \<longrightarrow>
   (gWls MOD s X \<longrightarrow> phi s X) \<and>
   (gWlsAbs MOD (us,s') A \<longrightarrow> phiAbs (us,s') A)"

lemma gInduct_elim:
assumes "gInduct MOD" and
Var: "\<And> xs x. phi (asSort xs) (gVar MOD xs x)" and
Op:
"\<And> delta inp' inp binp' binp.
    \<lbrakk>wlsInp delta inp'; wlsBinp delta binp'; gWlsInp MOD delta inp; gWlsBinp MOD delta binp;
     liftAll2 phi (arOf delta) inp; liftAll2 phiAbs (barOf delta) binp\<rbrakk>
    \<Longrightarrow> phi (stOf delta) (gOp MOD delta inp' inp binp' binp)" and
Abs:
"\<And> xs s x X' X.
   \<lbrakk>isInBar (xs,s); wls s X'; gWls MOD s X; phi s X\<rbrakk>
   \<Longrightarrow> phiAbs (xs,s) (gAbs MOD xs x X' X)"
shows
"(gWls MOD s X \<longrightarrow> phi s X) \<and>
 (gWlsAbs MOD (us,s') A \<longrightarrow> phiAbs (us,s') A)"
using assms unfolding gInduct_def apply -
apply(erule allE[of _ phi]) apply(erule allE[of _ phiAbs])
apply(erule allE[of _ s]) apply(erule allE[of _ X])
apply(erule allE[of _ us]) apply(erule allE[of _ s']) apply(erule allE[of _ A])
by blast

subsubsection {* Relevant predicates on maps from the term model *}

text{* Reflection of freshness *}

definition reflFresh where
"reflFresh h MOD ==
 \<forall> ys y s X.
   wls s X \<longrightarrow>
   gFresh MOD ys y X (h X) \<longrightarrow> fresh ys y X"

definition reflFreshAbs where
"reflFreshAbs hA MOD ==
 \<forall> ys y us s A.
   wlsAbs (us,s) A \<longrightarrow>
   gFreshAbs MOD ys y A (hA A) \<longrightarrow> freshAbs ys y A"

definition reflFreshAll where
"reflFreshAll h hA MOD ==
 reflFresh h MOD \<and> reflFreshAbs hA MOD"

lemmas reflFreshAll_defs = reflFreshAll_def
reflFresh_def reflFreshAbs_def

text{* Injectiveness *}

definition isInj where
"isInj h ==
 \<forall> s X Y.
   wls s X \<and> wls s Y \<longrightarrow>
   h X = h Y \<longrightarrow> X = Y"

definition isInjAbs where
"isInjAbs hA ==
 \<forall> us s A B.
   wlsAbs (us,s) A \<and> wlsAbs (us,s) B \<longrightarrow>
   hA A = hA B \<longrightarrow> A = B"

definition isInjAll where
"isInjAll h hA ==
 isInj h \<and> isInjAbs hA"

lemmas isInjAll_defs = isInjAll_def
isInj_def isInjAbs_def

text{* Surjectiveness *}

definition isSurj where
"isSurj h MOD ==
 \<forall> s X.
   gWls MOD s X \<longrightarrow>
   (\<exists> X'. wls s X' \<and> h X' = X)"

definition isSurjAbs where
"isSurjAbs hA MOD ==
 \<forall> us s A.
   gWlsAbs MOD (us,s) A \<longrightarrow>
   (\<exists> A'. wlsAbs (us,s) A' \<and> hA A' = A)"

definition isSurjAll where
"isSurjAll h hA MOD ==
 isSurj h MOD \<and> isSurjAbs hA MOD"

lemmas isSurjAll_defs = isSurjAll_def
isSurj_def isSurjAbs_def

subsubsection{* Criterion for the reflection of freshness *}

text{* First an auxiliary fact, independent of the type of model:  *}

lemma gFreshClsRev_recAll_reflFreshAll:
assumes pWls: "presWlsAll (rec MOD) (recAbs MOD) MOD"
and pCons: "presCons (rec MOD) (recAbs MOD) MOD"
and pFresh: "presFreshAll (rec MOD) (recAbs MOD) MOD"
and **: "gFreshClsRev MOD"
shows "reflFreshAll (rec MOD) (recAbs MOD) MOD"
proof-
  let ?h = "rec MOD"   let ?hA = "recAbs MOD"
  have pWlsInps[simp]:
  "!! delta inp. wlsInp delta inp \<Longrightarrow> gWlsInp MOD delta (lift ?h inp)"
  "!! delta binp. wlsBinp delta binp \<Longrightarrow> gWlsBinp MOD delta (lift ?hA binp)"
  using pWls presWls_wlsInp presWls_wlsBinp unfolding presWlsAll_def by auto
  {fix ys y s X us s' A
   have
   "(wls s X \<longrightarrow> gFresh MOD ys y X (rec MOD X) \<longrightarrow> fresh ys y X)
    \<and>
    (wlsAbs (us,s') A \<longrightarrow> gFreshAbs MOD ys y A (recAbs MOD A) \<longrightarrow> freshAbs ys y A)"
   apply(rule wls_induct) apply (tactic {* clarify_all_tac @{context} *})
   using assms unfolding presWlsAll_defs presCons_defs apply simp_all
   using ** unfolding gFreshClsRev_def gFreshGVarRev_def apply blast
   proof-
     fix delta inp binp
     let ?ar = "arOf delta"   let ?bar = "barOf delta"  let ?st = "stOf delta"
     assume inp: "wlsInp delta inp" and binp: "wlsBinp delta binp" and
     IH: "liftAll2 (\<lambda>s X. gFresh MOD ys y X (?h X) \<longrightarrow> fresh ys y X) ?ar inp" and
     BIH: "liftAll2 (\<lambda>s A. gFreshAbs MOD ys y A (?hA A) \<longrightarrow> freshAbs ys y A) ?bar binp"
     let ?linp = "lift ?h inp"  let ?lbinp = "lift ?hA binp"
     assume
     "gFresh MOD ys y
        (Op delta inp binp)
        (gOp MOD delta inp ?linp binp ?lbinp)"
     hence gFreshInps: "gFreshInp MOD ys y inp ?linp \<and> gFreshBinp MOD ys y binp ?lbinp"
     using inp binp ** unfolding gFreshClsRev_def gFreshGOpRev_def by force
     have freshInp: "freshInp ys y inp"
     unfolding freshInp_def liftAll_def proof clarify
       fix i X assume inpi: "inp i = Some X"
       then obtain s where "?ar i = Some s"
       using inp unfolding wlsInp_iff sameDom_def by(case_tac "?ar i", auto)
       moreover have "gFresh MOD ys y X (?h X)"
       using inpi gFreshInps unfolding gFreshInp_def liftAll2_def lift_def by fastforce
       ultimately show "fresh ys y X" using inpi IH unfolding liftAll2_def by auto
     qed
     have freshBinp: "freshBinp ys y binp"
     unfolding freshBinp_def liftAll_def proof clarify
       fix i A assume binpi: "binp i = Some A"
       then obtain us s where "?bar i = Some (us,s)"
       using binp unfolding wlsBinp_iff sameDom_def by(case_tac "?bar i", auto)
       moreover have "gFreshAbs MOD ys y A (?hA A)"
       using binpi gFreshInps unfolding gFreshBinp_def liftAll2_def lift_def by fastforce
       ultimately show "freshAbs ys y A" using binpi BIH unfolding liftAll2_def by auto
     qed
     show "freshInp ys y inp \<and> freshBinp ys y binp"
     using freshInp freshBinp by simp
   next
     fix s xs x X
     assume xs_s: "isInBar (xs,s)" and X: "wls s X"
     and "\<And>Y. \<lbrakk>wls s Y; skel Y = skel X\<rbrakk> \<Longrightarrow> gFresh MOD ys y Y (?h Y) \<longrightarrow> fresh ys y Y"
     hence IH: "gFresh MOD ys y X (?h X) \<longrightarrow> fresh ys y X" by simp
     have hX: "gWls MOD s (?h X)" using X pWls unfolding presWlsAll_defs by simp
     assume "gFreshAbs MOD ys y (Abs xs x X) (gAbs MOD xs x X (rec MOD X))" (*  (?hA (Abs xs x X)) *)
     moreover have "?hA (Abs xs x X) = gAbs MOD xs x X (?h X)"
     using xs_s X pCons unfolding presCons_defs by blast
     ultimately have 1: "gFreshAbs MOD ys y (Abs xs x X) (gAbs MOD xs x X (?h X))" by simp
     show "(ys = xs \<and> y = x) \<or> fresh ys y X"
     using xs_s hX X IH ** unfolding gFreshClsRev_def gFreshGAbsRev_def using 1 by force
   qed
  }
  thus ?thesis unfolding reflFreshAll_defs by blast
qed

text{* For fresh-swap models *}

theorem wlsFSw_recAll_reflFreshAll:
assumes "wlsFSw MOD" and "gFreshClsRev MOD"
shows "reflFreshAll (rec MOD) (recAbs MOD) MOD"
apply(rule gFreshClsRev_recAll_reflFreshAll)
using wlsFSw_recAll_termFSwMorph[of MOD]
unfolding termFSwMorph_def using assms by auto

text{* For fresh-subst models *}

theorem wlsFSb_recAll_reflFreshAll:
assumes "wlsFSb MOD" and "gFreshClsRev MOD"
shows "reflFreshAll (rec MOD) (recAbs MOD) MOD"
apply(rule gFreshClsRev_recAll_reflFreshAll)
using wlsFSb_recAll_termFSbMorph[of MOD]
unfolding termFSbMorph_def using assms by auto

(* Note: Here and below: No need for corresponding results for FSwSb and FSbSw models, as they
would follow at once from the above. *)

subsubsection{* Criterion for the injectiveness of the recursive map *}

text{* For fresh-swap models *}

theorem wlsFSw_recAll_isInjAll:
assumes *: "wlsFSw MOD"  "gAbsRenS MOD" and **: "gConsInj MOD"
shows "isInjAll (rec MOD) (recAbs MOD)"
proof-
  let ?h = "rec MOD"   let ?hA = "recAbs MOD"
  have 1: "termFSwMorph ?h ?hA MOD" using * wlsFSw_recAll_termFSwMorph by auto
  hence pWls: "presWlsAll ?h ?hA MOD"
  and pCons: "presCons ?h ?hA MOD"
  and pFresh: "presFreshAll ?h ?hA MOD"
  and pSwap: "presSwapAll ?h ?hA MOD" unfolding termFSwMorph_def by auto
  hence pWlsInps[simp]:
  "!! delta inp. wlsInp delta inp \<Longrightarrow> gWlsInp MOD delta (lift ?h inp)"
  "!! delta binp. wlsBinp delta binp \<Longrightarrow> gWlsBinp MOD delta (lift ?hA binp)"
  using presWls_wlsInp presWls_wlsBinp unfolding presWlsAll_def by auto
  {fix s X us s' A
   have
   "(wls s X \<longrightarrow> (\<forall> Y. wls s Y \<and> rec MOD X = rec MOD Y \<longrightarrow> X = Y))
    \<and>
    (wlsAbs (us,s') A \<longrightarrow> (\<forall> B. wlsAbs (us,s') B \<and> recAbs MOD A = recAbs MOD B \<longrightarrow> A = B))"
   apply(rule wls_induct) apply (tactic {* clarify_all_tac @{context} *})
   using 1 unfolding termFSwMorph_defs apply simp_all
   proof-
     fix xs x Y
     assume eq: "gVar MOD xs x = rec MOD Y" and Y: "wls (asSort xs) Y"
     show "Var xs x = Y"
     proof-
       {fix ys y assume Y_def: "Y = Var ys y" and "asSort ys = asSort xs"
        hence ys_def: "ys = xs" by simp
        have rec_y_def: "rec MOD (Var ys y) = gVar MOD ys y"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq ** unfolding Y_def rec_y_def gConsInj_def gVarInj_def
        unfolding ys_def by simp
       }
       moreover
       {fix delta1 inp1 binp1 assume inp1s: "wlsInp delta1 inp1"  "wlsBinp delta1 binp1"
        and Y_def: "Y = Op delta1 inp1 binp1" and st: "stOf delta1 = asSort xs"
        hence rec_Op_def:
        "rec MOD (Op delta1 inp1 binp1) =
         gOp MOD delta1 inp1 (lift ?h inp1) binp1 (lift ?hA binp1)"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq ** unfolding Y_def rec_Op_def gConsInj_def gVarGOpInj_def
        using inp1s st by simp
       }
       ultimately show ?thesis using wls_nchotomy[of "asSort xs" Y] Y by blast
     qed
   next
     fix delta inp binp Y
     assume inps: "wlsInp delta inp"  "wlsBinp delta binp" and Y: "wls (stOf delta) Y"
     and IH1: "liftAll2 (\<lambda>s X. \<forall>Y. wls s Y \<and> ?h X = ?h Y \<longrightarrow> X = Y) (arOf delta) inp"
     and IH2: "liftAll2 (\<lambda>us_s A. \<forall>B. wlsAbs us_s B \<and> ?hA A = ?hA B \<longrightarrow> A = B) (barOf delta) binp"
     and eq: "gOp MOD delta inp (lift ?h inp) binp (lift ?hA binp) = ?h Y"
     show "Op delta inp binp = Y"
     proof-
       {fix ys y assume Y_def: "Y = Var ys y" and st: "asSort ys = stOf delta"
        have rec_y_def: "rec MOD (Var ys y) = gVar MOD ys y"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq[THEN sym] ** unfolding Y_def rec_y_def gConsInj_def gVarGOpInj_def
        using inps st by simp
       }
       moreover
       {fix delta1 inp1 binp1 assume inp1s: "wlsInp delta1 inp1"  "wlsBinp delta1 binp1"
        and Y_def: "Y = Op delta1 inp1 binp1" and st: "stOf delta1 = stOf delta"
        hence rec_Op_def:
        "rec MOD (Op delta1 inp1 binp1) =
         gOp MOD delta1 inp1 (lift ?h inp1) binp1 (lift ?hA binp1)"
        using pCons unfolding presCons_defs by simp
        have 0: "delta = delta1 \<and> lift ?h inp = lift ?h inp1 \<and> lift ?hA binp = lift ?hA binp1"
        using eq ** unfolding Y_def rec_Op_def gConsInj_def gOpInj_def
        using inps inp1s st apply clarify
        apply(erule allE[of _ delta])  apply(erule allE[of _ delta1]) by force
        hence delta1_def: "delta1 = delta" by simp
        have 1: "inp = inp1"
        proof(rule ext)
          fix i
          show "inp i = inp1 i"
          proof(cases "inp i")
            assume Case1: "inp i = None"
            hence "lift ?h inp i = None" by(simp add: lift_None)
            hence "lift ?h inp1 i = None" using 0 by simp
            thus ?thesis unfolding Case1 by(simp add: lift_None)
          next
            fix X assume Case2: "inp i = Some X"
            hence "lift ?h inp i = Some (?h X)" unfolding lift_def by simp
            hence "lift ?h inp1 i = Some (?h X)" using 0 by simp
            then obtain Y where inp1_i: "inp1 i = Some Y" and hXY: "?h X = ?h Y"
            unfolding lift_def by(cases "inp1 i", auto)
            then obtain s where ar_i: "arOf delta i = Some s"
            using inp1s unfolding delta1_def wlsInp_iff sameDom_def
            by(case_tac "arOf delta i", auto)
            hence Y: "wls s Y"
            using inp1s inp1_i unfolding delta1_def wlsInp_iff liftAll2_def by auto
            thus ?thesis unfolding Case2 inp1_i apply simp
            using ar_i Case2 hXY IH1
            unfolding liftAll2_def by auto
          qed
        qed
        have 2: "binp = binp1"
        proof(rule ext)
          fix i
          show "binp i = binp1 i"
          proof(cases "binp i")
            assume Case1: "binp i = None"
            hence "lift ?hA binp i = None" by(simp add: lift_None)
            hence "lift ?hA binp1 i = None" using 0 by simp
            thus ?thesis unfolding Case1 by(simp add: lift_None)
          next
            fix A assume Case2: "binp i = Some A"
            hence "lift ?hA binp i = Some (?hA A)" unfolding lift_def by simp
            hence "lift ?hA binp1 i = Some (?hA A)" using 0 by simp
            then obtain B where binp1_i: "binp1 i = Some B" and hAB: "?hA A = ?hA B"
            unfolding lift_def by(cases "binp1 i", auto)
            then obtain us s where bar_i: "barOf delta i = Some (us,s)"
            using inp1s unfolding delta1_def wlsBinp_iff sameDom_def
            by(case_tac "barOf delta i", auto)
            hence B: "wlsAbs (us,s) B"
            using inp1s binp1_i unfolding delta1_def wlsBinp_iff liftAll2_def by auto
            thus ?thesis unfolding Case2 binp1_i apply simp
            using bar_i Case2 hAB IH2
            unfolding liftAll2_def by fastforce
          qed
        qed
        have ?thesis unfolding Y_def delta1_def 1 2 by simp
       }
       ultimately show ?thesis using wls_nchotomy[of "stOf delta" Y] Y by blast
     qed
   next
     fix s xs x X B
     assume xs_s: "isInBar (xs,s)" and X: "wls s X" and B: "wlsAbs (xs,s) B"
     and IH: "\<And>Xsw. (X,Xsw) \<in> swapped \<Longrightarrow> \<forall>Y. wls s Y \<and> ?h Xsw = ?h Y \<longrightarrow> Xsw = Y"
     and eq: "gAbs MOD xs x X (rec MOD X) = ?hA B" (* "?hA (Abs xs x X) = ?hA B" *)
     hence hX: "gWls MOD s (?h X)" using pWls unfolding presWlsAll_defs by simp
     show "Abs xs x X = B"
     proof-
       let ?P =
       "ParS
          (\<lambda> xs'. [])
          (\<lambda> s'. if s' = s then [X] else [])
          (\<lambda> us_s. [])
          []"
       have P: "wlsPar ?P" using X unfolding wlsPar_def by simp
       {fix y Y assume Y: "wls s Y" and B_def: "B = Abs xs y Y"
        hence hY: "gWls MOD s (?h Y)" using pWls unfolding presWlsAll_defs by simp
        let ?Xsw = "X #[y \<and> x]_xs"
        let ?hXsw = "gSwap MOD xs y x X (?h X)"
        have hXsw: "gWls MOD s ?hXsw"
        using X hX using * unfolding wlsFSw_def gSwapAllPresGWlsAll_defs by simp
        assume "\<forall> s. \<forall> Y \<in> termsOfS ?P s. fresh xs y Y"
        hence y_fresh: "fresh xs y X" by simp
        hence "gFresh MOD xs y X (?h X)"
        using X pFresh unfolding presFreshAll_defs by simp
        hence "gAbs MOD xs y (?Xsw) ?hXsw = gAbs MOD xs x X (?h X)"
        using xs_s X hX y_fresh * unfolding gAbsRenS_def by fastforce
        (* also have "\<dots> = gAbs MOD xs x X (rec MOD X)" (* ?hA (Abs xs x X) *)
        apply(rule sym) using xs_s X pCons xs_s unfolding presCons_defs by blast *)
        also have "\<dots> = ?hA B" using eq . 
        also have "recAbs MOD B = gAbs MOD xs y Y (?h Y)"          
        unfolding B_def using pCons xs_s Y unfolding presCons_defs by blast 
        finally have "gAbs MOD xs y ?Xsw ?hXsw = gAbs MOD xs y Y (?h Y)" .
        hence "?hXsw = ?h Y"
        using ** xs_s X hX hXsw Y hY unfolding gConsInj_def gAbsInj_def
        apply auto apply(erule allE[of _ xs]) apply(erule allE[of _ s])
        apply(erule allE[of _ y]) apply(erule allE[of _ ?Xsw]) by fastforce
        moreover have "?hXsw = ?h ?Xsw"
        using X pSwap unfolding presSwapAll_defs by simp
        ultimately have "?h ?Xsw = ?h Y" by simp
        moreover have "(X,?Xsw) \<in> swapped" using swap_swapped .
        ultimately have Y_def: "Y = ?Xsw" using Y IH by auto
        have ?thesis unfolding B_def Y_def
        using X y_fresh by simp
       }
       thus ?thesis using B P wlsAbs_fresh_nchotomy[of xs s B] by blast
     qed
   qed
  }
  thus ?thesis unfolding isInjAll_defs by blast
qed

text{* For fresh-subst models *}

theorem wlsFSb_recAll_isInjAll:
assumes *: "wlsFSb MOD" and **: "gConsInj MOD"
shows "isInjAll (rec MOD) (recAbs MOD)"
proof-
  let ?h = "rec MOD"   let ?hA = "recAbs MOD"
  have 1: "termFSbMorph ?h ?hA MOD" using * wlsFSb_recAll_termFSbMorph by auto
  hence pWls: "presWlsAll ?h ?hA MOD"
  and pCons: "presCons ?h ?hA MOD"
  and pFresh: "presFreshAll ?h ?hA MOD"
  and pSubst: "presSubstAll ?h ?hA MOD" unfolding termFSbMorph_def by auto
  hence pWlsInps[simp]:
  "!! delta inp. wlsInp delta inp \<Longrightarrow> gWlsInp MOD delta (lift ?h inp)"
  "!! delta binp. wlsBinp delta binp \<Longrightarrow> gWlsBinp MOD delta (lift ?hA binp)"
  using presWls_wlsInp presWls_wlsBinp unfolding presWlsAll_def by auto
  {fix s X us s' A
   have
   "(wls s X \<longrightarrow> (\<forall> Y. wls s Y \<and> rec MOD X = rec MOD Y \<longrightarrow> X = Y))
    \<and>
    (wlsAbs (us,s') A \<longrightarrow> (\<forall> B. wlsAbs (us,s') B \<and> recAbs MOD A = recAbs MOD B \<longrightarrow> A = B))"
   apply(rule wls_induct) apply (tactic {* clarify_all_tac @{context} *})
   using 1 unfolding termFSbMorph_defs apply simp_all
   proof-
     fix xs x Y
     assume eq: "gVar MOD xs x = rec MOD Y" and Y: "wls (asSort xs) Y"
     show "Var xs x = Y"
     proof-
       {fix ys y assume Y_def: "Y = Var ys y" and "asSort ys = asSort xs"
        hence ys_def: "ys = xs" by simp
        have rec_y_def: "rec MOD (Var ys y) = gVar MOD ys y"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq ** unfolding Y_def rec_y_def gConsInj_def gVarInj_def
        unfolding ys_def by simp
       }
       moreover
       {fix delta1 inp1 binp1 assume inp1s: "wlsInp delta1 inp1"  "wlsBinp delta1 binp1"
        and Y_def: "Y = Op delta1 inp1 binp1" and st: "stOf delta1 = asSort xs"
        hence rec_Op_def:
        "rec MOD (Op delta1 inp1 binp1) =
         gOp MOD delta1 inp1 (lift ?h inp1) binp1 (lift ?hA binp1)"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq ** unfolding Y_def rec_Op_def gConsInj_def gVarGOpInj_def
        using inp1s st by simp
       }
       ultimately show ?thesis using wls_nchotomy[of "asSort xs" Y] Y by blast
     qed
   next
     fix delta inp binp Y
     assume inps: "wlsInp delta inp"  "wlsBinp delta binp" and Y: "wls (stOf delta) Y"
     and IH1: "liftAll2 (\<lambda>s X. \<forall>Y. wls s Y \<and> ?h X = ?h Y \<longrightarrow> X = Y) (arOf delta) inp"
     and IH2: "liftAll2 (\<lambda>us_s A. \<forall>B. wlsAbs us_s B \<and> ?hA A = ?hA B \<longrightarrow> A = B) (barOf delta) binp"
     and eq: "gOp MOD delta inp (lift ?h inp) binp (lift ?hA binp) = ?h Y"
     show "Op delta inp binp = Y"
     proof-
       {fix ys y assume Y_def: "Y = Var ys y" and st: "asSort ys = stOf delta"
        have rec_y_def: "rec MOD (Var ys y) = gVar MOD ys y"
        using pCons unfolding presCons_defs by simp
        have ?thesis
        using eq[THEN sym] ** unfolding Y_def rec_y_def gConsInj_def gVarGOpInj_def
        using inps st by simp
       }
       moreover
       {fix delta1 inp1 binp1 assume inp1s: "wlsInp delta1 inp1"  "wlsBinp delta1 binp1"
        and Y_def: "Y = Op delta1 inp1 binp1" and st: "stOf delta1 = stOf delta"
        hence rec_Op_def:
        "rec MOD (Op delta1 inp1 binp1) =
         gOp MOD delta1 inp1 (lift ?h inp1) binp1 (lift ?hA binp1)"
        using pCons unfolding presCons_defs by simp
        have 0: "delta = delta1 \<and> lift ?h inp = lift ?h inp1 \<and> lift ?hA binp = lift ?hA binp1"
        using eq ** unfolding Y_def rec_Op_def gConsInj_def gOpInj_def
        using inps inp1s st apply clarify
        apply(erule allE[of _ delta])  apply(erule allE[of _ delta1]) by force
        hence delta1_def: "delta1 = delta" by simp
        have 1: "inp = inp1"
        proof(rule ext)
          fix i
          show "inp i = inp1 i"
          proof(cases "inp i")
            assume Case1: "inp i = None"
            hence "lift ?h inp i = None" by(simp add: lift_None)
            hence "lift ?h inp1 i = None" using 0 by simp
            thus ?thesis unfolding Case1 by(simp add: lift_None)
          next
            fix X assume Case2: "inp i = Some X"
            hence "lift ?h inp i = Some (?h X)" unfolding lift_def by simp
            hence "lift ?h inp1 i = Some (?h X)" using 0 by simp
            then obtain Y where inp1_i: "inp1 i = Some Y" and hXY: "?h X = ?h Y"
            unfolding lift_def by(cases "inp1 i", auto)
            then obtain s where ar_i: "arOf delta i = Some s"
            using inp1s unfolding delta1_def wlsInp_iff sameDom_def
            by(case_tac "arOf delta i", auto)
            hence Y: "wls s Y"
            using inp1s inp1_i unfolding delta1_def wlsInp_iff liftAll2_def by auto
            thus ?thesis unfolding Case2 inp1_i apply simp
            using ar_i Case2 hXY IH1
            unfolding liftAll2_def by auto
          qed
        qed
        have 2: "binp = binp1"
        proof(rule ext)
          fix i
          show "binp i = binp1 i"
          proof(cases "binp i")
            assume Case1: "binp i = None"
            hence "lift ?hA binp i = None" by(simp add: lift_None)
            hence "lift ?hA binp1 i = None" using 0 by simp
            thus ?thesis unfolding Case1 by(simp add: lift_None)
          next
            fix A assume Case2: "binp i = Some A"
            hence "lift ?hA binp i = Some (?hA A)" unfolding lift_def by simp
            hence "lift ?hA binp1 i = Some (?hA A)" using 0 by simp
            then obtain B where binp1_i: "binp1 i = Some B" and hAB: "?hA A = ?hA B"
            unfolding lift_def by(cases "binp1 i", auto)
            then obtain us s where bar_i: "barOf delta i = Some (us,s)"
            using inp1s unfolding delta1_def wlsBinp_iff sameDom_def
            by(case_tac "barOf delta i", auto)
            hence B: "wlsAbs (us,s) B"
            using inp1s binp1_i unfolding delta1_def wlsBinp_iff liftAll2_def by auto
            thus ?thesis unfolding Case2 binp1_i apply simp
            using bar_i Case2 hAB IH2
            unfolding liftAll2_def by fastforce
          qed
        qed
        have ?thesis unfolding Y_def delta1_def 1 2 by simp
       }
       ultimately show ?thesis using wls_nchotomy[of "stOf delta" Y] Y by blast
     qed
   next
     fix s xs x X B
     assume xs_s: "isInBar (xs,s)" and X: "wls s X" and B: "wlsAbs (xs,s) B"
     and IH: "\<And>ys y1 y2. \<forall>Y. wls s Y \<and> rec MOD (X #[y1 // y2]_ys) = rec MOD Y \<longrightarrow> X #[y1 // y2]_ys = Y"
     and eq: "gAbs MOD xs x X (rec MOD X) = ?hA B"
     hence hX: "gWls MOD s (?h X)" using pWls unfolding presWlsAll_defs by simp
     show "Abs xs x X = B"
     proof-
       let ?P =
       "ParS
          (\<lambda> xs'. [])
          (\<lambda> s'. if s' = s then [X] else [])
          (\<lambda> us_s. [])
          []"
       have P: "wlsPar ?P" using X unfolding wlsPar_def by simp
       {fix y Y assume Y: "wls s Y" and B_def: "B = Abs xs y Y"
        hence hY: "gWls MOD s (?h Y)" using pWls unfolding presWlsAll_defs by simp
        let ?Xsb = "X #[y // x]_xs"
        let ?hXsb = "gSubst MOD xs (Var xs y) (gVar MOD xs y) x X (?h X)"
        have 1: "wls (asSort xs) (Var xs y) \<and> gWls MOD (asSort xs) (gVar MOD xs y)"
        using * unfolding wlsFSb_def gConsPresGWls_defs by simp
        hence hXsb: "gWls MOD s ?hXsb"
        using X hX using * unfolding wlsFSb_def gSubstAllPresGWlsAll_defs by simp
        assume "\<forall> s. \<forall> Y \<in> termsOfS ?P s. fresh xs y Y"
        hence y_fresh: "fresh xs y X" by simp
        hence "gFresh MOD xs y X (?h X)"
        using X pFresh unfolding presFreshAll_defs by simp
        hence "gAbs MOD xs y (?Xsb) ?hXsb = gAbs MOD xs x X (?h X)"
        using xs_s X hX y_fresh * unfolding wlsFSb_def gAbsRen_def by fastforce
        also have "\<dots> = ?hA B" using eq .
        also have "\<dots> = gAbs MOD xs y Y (?h Y)"
        unfolding B_def using pCons xs_s Y unfolding presCons_defs by blast
        finally have
        "gAbs MOD xs y ?Xsb ?hXsb = gAbs MOD xs y Y (?h Y)" .
        hence "?hXsb = ?h Y"
        using ** xs_s X hX hXsb Y hY unfolding gConsInj_def gAbsInj_def
        apply auto apply(erule allE[of _ xs]) apply(erule allE[of _ s])
        apply(erule allE[of _ y]) apply(erule allE[of _ ?Xsb]) by fastforce
        moreover have "?hXsb = ?h ?Xsb"
        using X pSubst 1 pCons unfolding presSubstAll_defs vsubst_def presCons_defs by simp
        ultimately have "?h ?Xsb = ?h Y" by simp
        hence Y_def: "Y = ?Xsb" using Y IH by auto
        have ?thesis unfolding B_def Y_def
        using X y_fresh by simp
       }
       thus ?thesis using B P wlsAbs_fresh_nchotomy[of xs s B] by blast
     qed
   qed
  }
  thus ?thesis unfolding isInjAll_defs by blast
qed

subsubsection{* Criterion for the surjectiveness of the recursive map *}

text{* First an auxiliary fact, independent of the type of model:  *}

lemma gInduct_gConsIndif_recAll_isSurjAll:
assumes pWls: "presWlsAll (rec MOD) (recAbs MOD) MOD"
and pCons: "presCons (rec MOD) (recAbs MOD) MOD"
and "gConsIndif MOD" and *: "gInduct MOD"
shows "isSurjAll (rec MOD) (recAbs MOD) MOD"
proof-
  let ?h = "rec MOD"   let ?hA = "recAbs MOD"
  {fix s X us s' A
   have
   "(gWls MOD s X \<longrightarrow> (\<exists> X'. wls s X' \<and> rec MOD X' = X))
    \<and>
    (gWlsAbs MOD (us,s') A \<longrightarrow> (\<exists> A'. wlsAbs (us,s') A' \<and> recAbs MOD A' = A))"
   using * apply -
   apply(erule gInduct_elim) apply auto
   proof-
     fix xs x
     show "\<exists>X'. wls (asSort xs) X' \<and> rec MOD X' = gVar MOD xs x"
     apply(intro exI[of _ "Var xs x"])
     using pWls pCons unfolding presWlsAll_defs presCons_defs by simp
   next
     fix delta inp' inp binp' binp
     let ?ar = "arOf delta"   let ?bar = "barOf delta"  let ?st = "stOf delta"
     assume inp': "wlsInp delta inp'" and binp': "wlsBinp delta binp'"
     and inp: "gWlsInp MOD delta inp" and binp: "gWlsBinp MOD delta binp"
     and IH: "liftAll2 (\<lambda>s X. \<exists>X'. wls s X' \<and> ?h X' = X) ?ar inp"
     and BIH: "liftAll2 (\<lambda>us_s A. \<exists>A'. wlsAbs us_s A' \<and> ?hA A' = A) ?bar binp"
     (*  *)
     let ?phi = "\<lambda> s X X'. wls s X' \<and> ?h X' = X"
     obtain inp1' where inp1'_def:
     "inp1' =
      (\<lambda> i.
         case (?ar i, inp i) of
           (None, None) \<Rightarrow> None
          |(Some s, Some X) \<Rightarrow> Some (SOME X'. ?phi s X X'))" by blast
     hence [simp]:
     "\<And> i. ?ar i = None \<and> inp i = None \<Longrightarrow> inp1' i = None"
     "\<And> i s X. ?ar i = Some s \<and> inp i = Some X \<Longrightarrow> inp1' i = Some (SOME X'. ?phi s X X')"
     unfolding inp1'_def by auto
     have inp1': "wlsInp delta inp1'"
     using inp' unfolding wlsInp_iff apply auto
     proof-
       show "sameDom ?ar inp1'"
       unfolding sameDom_def proof clarify
         fix i
         have "(?ar i = None) = (inp i = None)"
         using inp unfolding gWlsInp_def sameDom_def by simp
         thus "(?ar i = None) = (inp1' i = None)"
         unfolding inp1'_def by auto
       qed
     next
       show "liftAll2 wls ?ar inp1'"
       unfolding liftAll2_def proof auto
         fix i s X1'
         assume ari: "?ar i = Some s" and inp1'i: "inp1' i = Some X1'"
         have "sameDom inp ?ar"
         using inp unfolding gWlsInp_def using sameDom_sym by blast
         then obtain X where inpi: "inp i = Some X"
         using ari unfolding sameDom_def by(cases "inp i", auto)
         hence X1'_def: "X1' = (SOME X1'. ?phi s X X1')"
         using ari inp1'i unfolding inp1'_def by simp
         obtain X' where X': "?phi s X X'"
         using inpi ari IH unfolding liftAll2_def by blast
         hence "?phi s X X1'"
         unfolding X1'_def by(rule someI[of "?phi s X"])
         thus "wls s X1'" by simp
       qed
     qed
     have lift_inp1': "lift ?h inp1' = inp"
     proof(rule ext)
       fix i let ?linp1' = "lift ?h inp1'"
       show "?linp1' i = inp i"
       apply(case_tac "inp i", auto)
       proof-
         fix i assume inpi: "inp i = None"
         hence "?ar i = None" using inp unfolding gWlsInp_def sameDom_def by simp
         hence "inp1' i = None" using inpi by simp
         thus "?linp1' i = None" unfolding lift_def by simp
       next
         fix X
         assume inpi: "inp i = Some X"
         then obtain s where ari: "?ar i = Some s"
         using inp unfolding gWlsInp_def sameDom_def by(cases "?ar i", auto)
         let ?X1' = "SOME X1'. ?phi s X X1'"
         have inp1'i: "inp1' i = Some ?X1'" using ari inpi by simp
         hence linp1'i: "?linp1' i = Some (?h ?X1')" unfolding lift_def by simp
         obtain X' where X': "?phi s X X'"
         using inpi ari IH unfolding liftAll2_def by blast
         hence "?phi s X ?X1'" by(rule someI[of "?phi s X"])
         thus "?linp1' i = Some X" using linp1'i by simp
       qed
     qed
     (*  *)
     let ?bphi = "\<lambda> (us,s) A A'. wlsAbs (us,s) A' \<and> ?hA A' = A"
     obtain binp1' where binp1'_def:
     "binp1' =
      (\<lambda> i.
         case (?bar i, binp i) of
           (None, None) \<Rightarrow> None
          |(Some (us,s), Some A) \<Rightarrow> Some (SOME A'. ?bphi (us,s) A A'))" by blast
     hence [simp]:
     "\<And> i. ?bar i = None \<and> binp i = None \<Longrightarrow> binp1' i = None"
     and *:
     "\<And> i us s A. ?bar i = Some (us,s) \<and> binp i = Some A \<Longrightarrow>
        binp1' i = Some (SOME A'. ?bphi (us,s) A A')"
     unfolding binp1'_def by auto
     have binp1': "wlsBinp delta binp1'"
     using binp' unfolding wlsBinp_iff apply auto
     proof-
       show "sameDom ?bar binp1'"
       unfolding sameDom_def proof clarify
         fix i
         have "(?bar i = None) = (binp i = None)"
         using binp unfolding gWlsBinp_def sameDom_def by simp
         thus "(?bar i = None) = (binp1' i = None)"
         unfolding binp1'_def by auto
       qed
     next
       show "liftAll2 wlsAbs ?bar binp1'"
       unfolding liftAll2_def proof auto
         fix i us s A1'
         assume bari: "?bar i = Some (us,s)" and binp1'i: "binp1' i = Some A1'"
         have "sameDom binp ?bar"
         using binp unfolding gWlsBinp_def using sameDom_sym by blast
         then obtain A where binpi: "binp i = Some A"
         using bari unfolding sameDom_def by(cases "binp i", auto)
         hence A1'_def: "A1' = (SOME A1'. ?bphi (us,s) A A1')"
         using bari binp1'i unfolding binp1'_def by simp
         obtain A' where A': "?bphi (us,s) A A'"
         using binpi bari BIH unfolding liftAll2_def by fastforce
         hence "?bphi (us,s) A A1'"
         unfolding A1'_def by(rule someI[of "?bphi (us,s) A"])
         thus "wlsAbs (us,s) A1'" by simp
       qed
     qed
     have lift_binp1': "lift ?hA binp1' = binp"
     proof(rule ext)
       fix i let ?lbinp1' = "lift ?hA binp1'"
       show "?lbinp1' i = binp i"
       apply(case_tac "binp i", auto)
       proof-
         fix i assume binpi: "binp i = None"
         hence "?bar i = None" using binp unfolding gWlsBinp_def sameDom_def by simp
         hence "binp1' i = None" using binpi by simp
         thus "?lbinp1' i = None" unfolding lift_def by simp
       next
         fix A
         assume binpi: "binp i = Some A"
         then obtain us s where bari: "?bar i = Some (us,s)"
         using binp unfolding gWlsBinp_def sameDom_def by(cases "?bar i", auto)
         let ?A1' = "SOME A1'. ?bphi (us,s) A A1'"
         have binp1'i: "binp1' i = Some ?A1'" using bari binpi *[of i us s A] by simp
         hence lbinp1'i: "?lbinp1' i = Some (?hA ?A1')" unfolding lift_def by simp
         obtain A' where A': "?bphi (us,s) A A'"
         using binpi bari BIH unfolding liftAll2_def by fastforce
         hence "?bphi (us,s) A ?A1'" by(rule someI[of "?bphi (us,s) A"])
         thus "?lbinp1' i = Some A" using lbinp1'i by simp
       qed
     qed
     (*  *)
     let ?X' = "Op delta inp1' binp1'"
     have X': "wls ?st ?X'" using inp1' binp1' by simp
     have "?h ?X' = gOp MOD delta inp1' inp binp1' binp"
     using inp1' binp1' pCons lift_inp1' lift_binp1'
     unfolding presCons_defs by simp
     hence "?h ?X' = gOp MOD delta inp' inp binp' binp"
     using inp' inp1' inp binp' binp1' binp assms
     unfolding gConsIndif_defs by metis
     thus "\<exists>X'. wls (stOf delta) X' \<and> ?h X' = gOp MOD delta inp' inp binp' binp"
     using X' by blast
   next
     fix xs s x X' X1'
     assume xs_s: "isInBar (xs,s)" and X': "wls s X'" and
     hX1': "gWls MOD s (?h X1')" and X1': "wls s X1'"
     thus "\<exists>A'. wlsAbs (xs,s) A' \<and> ?hA A' = gAbs MOD xs x X' (?h X1')"
     apply - apply(intro exI[of _ "Abs xs x X1'"]) apply auto
     using pCons unfolding presCons_def presAbs_def apply auto
     apply(elim allE[of _ xs]) apply(elim allE[of _ x]) apply(elim allE[of _ s]) apply simp
     using assms unfolding gConsIndif_defs by blast
   qed
  }
  thus ?thesis unfolding isSurjAll_defs by blast
qed

text{* For fresh-swap models *}

theorem wlsFSw_recAll_isSurjAll:
assumes "wlsFSw MOD" and "gConsIndif MOD" and "gInduct MOD"
shows "isSurjAll (rec MOD) (recAbs MOD) MOD"
apply(rule gInduct_gConsIndif_recAll_isSurjAll)
using wlsFSw_recAll_termFSwMorph[of MOD]
unfolding termFSwMorph_def using assms by auto

text{* For fresh-subst models *}

theorem wlsFSb_recAll_isSurjAll:
assumes "wlsFSb MOD" and "gConsIndif MOD" and "gInduct MOD"
shows "isSurjAll (rec MOD) (recAbs MOD) MOD"
apply(rule gInduct_gConsIndif_recAll_isSurjAll)
using wlsFSb_recAll_termFSbMorph[of MOD]
unfolding termFSbMorph_def using assms by auto

(********************************************)
lemmas recursion_simps =
fromMOD_simps
ipresCons_fromMOD_fst_all_simps
fromIMor_simps

declare recursion_simps [simp del]

end (* context FixSyn *)
(************************************************)

end
